/** -*- mode: C++ -*-
 * search.h defines the alpha-beta minimax algorithm
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#ifndef SEARCH_H
#define SEARCH_H
#include "generate.h"
#include <fstream>

class HashEntry;
enum BoundType { EXACT=0, UPPERBOUND=1, LOWERBOUND=2 };  // only two bits needed

extern jlong        pieceSquareTable[144][312]; // 12 files * 12 ranks * 39 pieces * 2 colours,etc 
extern jint checksumPieceSquareTable[144][312]; // 12 files * 12 ranks * 39 pieces * 2 colours,etc 
extern jshort                   hashMultiplier; // how many duplicate slots for a given hashcode
extern jlong                          hashBits; // how many bits of the hashcode to us as an index 
extern HashEntry                    *hashTable; // address of the hashtable
extern bool timeExhausted;

class Search {
  static const unsigned int KILLER_LIST_SIZE=7;
  static const jint ILLEGAL=-2000000;
public:
  static const jint INFINITY=1000000;
  static const jint MINUSINFINITY=-1000000;
  static const jint MAXIMUMDEPTH=64;
  static jint cached;
  ofstream           logFile;
private:
  long                 idNum;    // used for trace nodes
  bool            matingMode;    // true == look for mate
  bool             notProven;    // true == no mate found at maximumDepth
  bool                   log;    // debuging option
  // the level of logging depends upon the detailed variable
  // this variable is a bit-mask, with the following fields:
  //
  // 3      2       1 
  // 1      4       6       8       0 
  // ................................
  //                         ||||||||
  //                         |||||||transpositions (hits)
  //                         ||||||repetitions
  //                         |||||transpositions (stores)
  //                         ||||trace calls to search   
  //                         |||avoiding checks
  //                         ||failing to give check 
  //                         |mating mode analysis
  //                         trace alpha cutoffs                             
  jint              detailed;    
  bool     transpositionHits;
  bool        logRepetitions;
  bool                stores;
  bool                 trace;
  bool           avoidsCheck;
  bool            failsCheck;
  bool        matingAnalysis;
  bool          alphaCutoffs;
  bool                colour;    // the computer's colour
  jint          maximumDepth;
  jint         greatestDepth;    // for stats logging
  jint                 count;    // nodes searched
  jint             exactHits;    // Transition table hits (exact score)
  jint             upperHits;    // Transition table hits (upper bound)
  jint             lowerHits;    // Transition table hits (lower bound)
  jint          killerMisses;    // Transition table near-misses (add to the killer list)
  jint            collisions;    // Transition table hashing collisions
  jint          replacements;    // Transition table replacements
  jint            hashErrors;    // Transition table hashing errors
  jint       repetitionCount;    // how many times an illegal position is generated
  GeneratedMove   *firstMove;
  Board           *homeBoard;
  bool         alwaysPromote;
  bool           repetitions;

  killerListType killerList[30];
  Move principalContinuation[MAXIMUMDEPTH][MAXIMUMDEPTH];

public:

  Search(GeneratedMove *fm,Board*hb,bool c, bool a, bool r, bool l, jint d, bool m) {
    firstMove=fm;
    homeBoard=hb;
    colour=c;
    maximumDepth=0;
    greatestDepth=0;
    alwaysPromote=a;
    count=0;
    exactHits=0;
    lowerHits=0;
    upperHits=0;
    killerMisses=0;
    collisions=0;
    hashErrors=0;
    replacements=0;
    repetitionCount=0;
    repetitions=r;
    log=l;
    matingMode=m;
    notProven=false;
    detailed=d;
    transpositionHits=detailed & 0x00000001;    
    logRepetitions=detailed & 0x00000002;
    stores=detailed & 0x00000004;
    trace=detailed & 0x00000008;
    avoidsCheck=detailed & 0x00000010;
    failsCheck=detailed & 0x00000020;
    matingAnalysis=detailed & 0x00000040;
    alphaCutoffs=detailed & 0x00000080;
    idNum=0;

    for(jint i=0;i<maximumDepth;i++) 
      for(jint j=0;j<maximumDepth;j++) 
	principalContinuation[j][i].invalidate();

  }
  void setMaxDepth(jint m) { 
    maximumDepth=(m > MAXIMUMDEPTH ? MAXIMUMDEPTH : m);
  }
  jint getGreatestDepth() const { return greatestDepth; }
  void updatePrincipalContinuation(jint d, const Move& m)
  { 
    principalContinuation[d][d]=m;
    for(jint i=d+1;i<maximumDepth;i++) {
    principalContinuation[d][i]=principalContinuation[d+1][i];
    }
  }
  void invalidatePrincipalContinuation(jint d)
  {
    principalContinuation[d][d].invalidate();
  }
  Move getPrincipalContinuation(jint i) {
    return principalContinuation[0][i];
  }

  void store(jint, jlong, jint, jint, jint, BoundType, const Move&);
  void logStore(BoundType, jint, const Move&, bool);
  jint search(jint,jint,bool,bool,jint,jint,GeneratedMove&, jint, jlong, jint, bool);
  Move&   getBestMove() { return principalContinuation[0][0]; }
  jint    getNodesSearched() { return count; }
  jint    getExactHits() { return exactHits; }
  jint    getUpperCutoffs() { return upperHits; }
  jint    getLowerCutoffs() { return lowerHits; }
  jint    getNearMisses() { return killerMisses; }
  jint    getHashingCollisions() { return collisions; }
  jint    getHashingErrors() { return hashErrors; }
  jint    getReplacements() { return replacements; }
  jint    getRepetitions() { return repetitionCount; }
  void openLogFile(const char *c) { 
    logFile.open(c); 
    logFile << "<?xml version=\"1.0\" standalone='yes'?>" << endl;
  }
  void closeLogFile() { logFile.flush(); logFile.close(); }
  bool wasNotProven() { return notProven; }
};

class HashEntry {
  jlong             hashcode; // 64-bit hash code
  jint              checksum; // 32-bit checksum
  jint                 value; // this position's score 
  signed         height : 21; // limits us to 2**20 ply searches
  unsigned         bound : 2;
  unsigned        source : 8; // these fields encode the best move
  unsigned   destination : 8; // from this position
  unsigned  destination2 : 8;
  unsigned       squares : 1;
  unsigned     promotion : 1;
  unsigned  firstCapture : 1;
  unsigned secondCapture : 1;
  unsigned        abbrev : 6; // useful for reconstructing the Move
  unsigned        colour : 1; // this is a lookup-only field

  static jbyte pack(const Coordinates &c) {
    jbyte X=jbyte(c.x()) << 4;
    return (X+jbyte(c.y()));
  }
public:
  HashEntry() { hashcode=0; checksum=0; }
  HashEntry(jlong hc, jint ck, jint val, jint h, BoundType bt, Move m) {
    hashcode=hc; checksum=ck; value=val; height=h; bound=bt;
    
    if (m.isBlack()) colour = 1;
    else colour=0;
    source=pack(m.getSource());
    destination=pack(m.getStage1());
    squares=m.squares()-1;
    if (squares==1) destination2=pack(m.getStage2());
    if (m.wasPromoted()) promotion = 1;
    else promotion = 0;
    if (m.wasFirstCaptured()) firstCapture=1;
    else firstCapture=0;
    if (m.wasSecondCaptured()) secondCapture=1;
    else secondCapture=0;
    abbrev=m.getAbbrev();
  }
  jlong getHashcode() const { return hashcode; }
  jint getChecksum() const { return checksum; }
  jint getValue() const { return value; }
  jint isBlack() const { return colour; }
  jint getHeight() const { return height; }
  BoundType getBound() const { return BoundType(bound); }
  void setMove(Move &m) {
    m.setColour(colour == 1);
    m.setNumber(0);    
    m.setAbbrev(PieceType(abbrev));
    m.setSquares(squares+1);
    m.setSource(Coordinates(source >> 4, source & 0x0f ));
    m.setDestination1(Coordinates(destination >> 4, destination  & 0x0f));
    if (squares == 1) m.setDestination2(Coordinates(destination2 >> 4, destination2  & 0x0f));
    if (promotion == 1) m.setPromoted();
    if (firstCapture == 1) m.setFirstCapture();
    if (secondCapture == 1) m.setSecondCapture();
  }
};

jlong hash(const Board &);
jlong adjustHash(jlong,  const Board&, const Move &);
jint  checksum(const Board &);
jint  adjustChecksum(jint,  const Board&, const Move &);
jint  randomNumber(jint);

bool operator<(const pair<GeneratedMove,jint> &, const pair<GeneratedMove,jint> &);
#endif
