/**
 * search.cpp implements the alpha-beta negamax algorithm
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

using namespace std;

#ifdef INLINE_OK 
#define INLINE inline
#else
#define INLINE  
#endif
#include "search.h"
#include <strstream.h>
#include <algorithm>
#include <functional>


bool operator<(const pair<GeneratedMove,jint> &x, const pair<GeneratedMove,jint> &y) {
  return x.second > y.second; // we want the list sorted in descending order of gain
}

INLINE jint Search::search(jint depth,jint maxDepth,bool extraDeep, bool extraDeepPending,
			   jint alpha,jint beta,GeneratedMove& gm, jint gain,
			   jlong hashcode, jint checkSum, bool onPC)
{

  if (timeExhausted) {
      if (log && trace) logFile << "<trace/>" << endl;
      return 0; // doesn't matter what we return
  }

  if (depth >= maxDepth-1) onPC = false; // can't become true again
  jlong newHashcode=hashcode;
  jint newChecksum=checkSum;

  count+=1;
  if (greatestDepth < depth) greatestDepth = depth;

  bool even=(depth % 2) == 0;
  Move thisMove=gm.getMove();

  // force repetition checking for Black (the computer) in mating mode

  bool checkRepetitions = repetitions;
  if (matingMode) {
    if (even) checkRepetitions = false;
    else checkRepetitions = true;
  }

  if (log && trace) {
    if ( depth > 0 ) {
      char buffer[30];
      thisMove.notate(buffer); 
      logFile << "<trace id=\"" << idNum++ << "\" depth=\"" << depth << "\">" 
	      << buffer << endl;
    }
    else logFile << "<trace  id=\"" << idNum++ <<  "\" depth=\"" << depth << "\">" << endl;
  }

  bool capture=thisMove.wasFirstCaptured();
  jint squares=thisMove.squares();
  bool secondCapture=thisMove.wasSecondCaptured();

  jint thisMaxDepth=maxDepth;
  if (!matingMode && depth == maxDepth && capture) {
    if (squares == 2 && !secondCapture && extraDeep) ; // gets too tricky
    else {
      if ( thisMaxDepth < MAXIMUMDEPTH ) thisMaxDepth+=1;
      extraDeepPending=true;
    }
  }

  if (depth > 0 && !gm.evaluateNow()) {
    if (depth < thisMaxDepth) {
      newHashcode=adjustHash(hashcode, *homeBoard, thisMove);
      newChecksum=adjustChecksum(checkSum, *homeBoard, thisMove);
    }
    gm.updatePosition(*homeBoard, checkRepetitions);
    // updatePosition will call restorePosition if the move proves not to be legal 
  }

  if (depth > 0 && !gm.isLegal()) {
    if (log && logRepetitions) {
      char buffer[30];
      thisMove.notate(buffer);
      logFile << "<repetiton>" << buffer << "</repetiton>" << endl;
    }
    if (!matingMode) {
      if (log && trace) {
	logFile << "<value>" << ILLEGAL << "</value>" << endl;
	logFile << "</trace>" << endl;
      }
      return ILLEGAL;  // special flag  
    }
    else {
      if (log && trace) {
	logFile << "<value>" << (even ? maximumDepth-depth+1 : 1) << "</value>" << endl;
	logFile << "</trace>" << endl;
      }
      return(even ? maximumDepth-depth+1 : 1); // disallowed
    }
  }

  if (matingMode && (even ? homeBoard->isCheck(colour) : !homeBoard->isCheck(colour))) {
    if (log && ((even && avoidsCheck) || (!even && failsCheck))) {
      char buffer[30];
      thisMove.notate(buffer);
      if (even) logFile << "<avoidCheck>" << buffer << "</avoidCheck>" << endl;
      else logFile << "<notCheck>" << buffer << "</notCheck>" << endl;
    }
    if (depth > 0) gm.restorePosition(*homeBoard, true);
    if (log && trace) {
      logFile << "<value>" << (even ? maximumDepth-depth+1 : 1) << "</value>" << endl;
      logFile << "</trace>" << endl;
    }
    return (even ? maximumDepth-depth+1 : 1); // failed to avoid/give check
  }
  jint hashIndex=jint(newHashcode & hashBits);

  bool lateEvaluateNow=false;
  if (!matingMode && homeBoard->getCrownPrinces(true) == 0 
      || homeBoard->getCrownPrinces(false) == 0) {
    gm.setEvaluateNow(); // don't bother searching after end-of-game (ignores bare king)
    lateEvaluateNow=true;
  }
  bool leaf=(depth == thisMaxDepth || gm.evaluateNow());

  /* now check the transposition table */
  if (depth >0 && !matingMode && newHashcode != 0) { 
    int multiplier=1;
    bool keepTrying=true;
    while (keepTrying && multiplier <= hashMultiplier) {
      HashEntry hashEntry=hashTable[hashIndex*multiplier];
      if (hashEntry.getHashcode() == newHashcode
	  && hashEntry.isBlack() != thisMove.isBlack() 
	  && hashEntry.getChecksum() == newChecksum) {
	
	jint value=hashEntry.getValue();
	jint foundHeight=hashEntry.getHeight();
        jint height=maximumDepth-depth;
	if (matingMode && height != foundHeight) 
	  value+=foundHeight-height; 
	Move recommendation;
	hashEntry.setMove(recommendation); // at the very least, it will go on the killer list
	
	if (foundHeight >= height) 
	  switch (hashEntry.getBound()) { 
	    // otherwise, just a recommendation
	  case EXACT: 
	    exactHits+=1;
	    if (depth > 0) gm.restorePosition(*homeBoard, checkRepetitions);
	    if (log && transpositionHits) {
	      char buffer[30];
	      recommendation.notate(buffer);
	      if (matingMode && value == -1) 
		logFile << "<transpose bound=\"exact\" mate=\"avoid\">" << buffer 
			<< "</transpose>" << endl;
	      else if (matingMode) logFile << "<transpose bound=\"exact\" mate=\"yes\" value=\"" 
					   << value << "\">" << buffer << "</transpose>" <<endl; 
	      else logFile  << "<transpose bound=\"exact\" value=\"" << value 
			    << "\">" << buffer << "</transpose>" << endl; 
	    }
	    if (log && trace) {
	      logFile << "<value>" << value << "</value>" << endl;
	      logFile << "</trace>" << endl;
	    }
            invalidatePrincipalContinuation(depth);
	    return value; // a hit!
	    break;
	  case UPPERBOUND:
	    if (value <= alpha) {
	      upperHits+=1;
	      if (depth > 0) gm.restorePosition(*homeBoard, checkRepetitions);
	      if (log && transpositionHits) {
		char buffer[30];
		recommendation.notate(buffer);
		if (matingMode && value == -1) 
		  logFile << "<transpose bound=\"upper\" mate=\"avoid\">" << buffer 
			  << "</transpose>" << endl;
		else if (matingMode) logFile << "<transpose bound=\"upper\" mate=\"yes\" value=\"" 
					     << value << "\">" << buffer << "</transpose>" <<endl; 
		else logFile  << "<transpose bound=\"upper\" value=\"" << value 
			      << "\">" << buffer << "</transpose>" << endl; 
	      }
	      if (log && trace) {
		logFile << "<value>" << value << "</value>" << endl;
		logFile << "</trace>" << endl;
	      }
	      invalidatePrincipalContinuation(depth);
	      return value; // good enough
	    }
	    break;
	  case LOWERBOUND:
	    if (value >= beta) {
	      lowerHits+=1;
	      if (depth > 0) gm.restorePosition(*homeBoard, checkRepetitions);
	      if (log && transpositionHits) {
		char buffer[30];
		recommendation.notate(buffer);
		if (matingMode && value == -1) 
		  logFile << "<transpose bound=\"lower\" mate=\"avoid\">" << buffer 
			  << "</transpose>" << endl;
		else if (matingMode) logFile << "<transpose bound=\"lower\" mate=\"yes\" value=\"" 
					     << value << "\">" << buffer << "</transpose>" <<endl; 
		else logFile  << "<transpose bound=\"lower\" value=\"" << value 
			      << "\">" << buffer << "</transpose>" << endl; 
	      }
	      if (log && trace) {
		logFile << "<value>" << value << "</value>" << endl;
		logFile << "</trace>" << endl;
	      }
	      invalidatePrincipalContinuation(depth);
	      return value;  // good enough
	    }
	    break;
	  }
	// if we reach here, then the hash hit wasn't sufficient for a cut-off, but still
	// we add the recomended move to the back of the killer list
	if (!matingMode && !leaf && hashEntry.isBlack() != thisMove.isBlack()) {
	  killerMisses+=1;
	  // keep the killer list short
	  if (killerList[depth].size() > KILLER_LIST_SIZE-1) killerList[depth].pop_back(); 
	  killerList[depth].push_back(recommendation);
	}
	keepTrying=false;
      }
      else {
	if (hashEntry.getHashcode() == 0) keepTrying=false; // never-used slot
	else multiplier+=1;
      }
    }
  }
  
  if (leaf) {
    if (matingMode) {
      if (log && matingAnalysis) {
	char buffer[30];
	thisMove.notate(buffer);
	logFile << "<maxDepth>" << buffer << "</maxDepth>" << endl;
      }
      gm.restorePosition(*homeBoard, true);
      notProven=true;
      if (log && trace) {
	logFile << "<value>" << (even ? 0 : maximumDepth) << "</value>" << endl;
	logFile << "</trace>" << endl;
      }
      return (even ? 0 : maximumDepth ); // no mate found
    }

    jint thisScore=homeBoard->evaluate(colour);
    if (!even) thisScore = -thisScore; 
    // the evaluateNow node will be restored by it's parent-copy
    if (!gm.evaluateNow() || lateEvaluateNow) gm.restorePosition(*homeBoard, checkRepetitions);
    if (log && trace) {
      logFile << "<value>" << thisScore << "</value>" << endl;
      logFile << "</trace>" << endl;
    }
    return thisScore;
  }
  else if (matingMode && log && matingAnalysis) { // must be a check, or an evasion
    char buffer[30];
    thisMove.notate(buffer);
    if (depth == 0);
    else if (even) logFile << "<evade>" << buffer << endl;
    else logFile << "<check>" << buffer << endl;
  }

  // we have a branch node
  
  bool exactMatch=false;
  killerListType*killers=(matingMode ? NULL : &killerList[depth]); 
  Generator generator=Generator(homeBoard,&thisMove,extraDeep,alwaysPromote,killers);
  generator.generate();

  listType &listOfMoves=generator.getMoves();
  listOfMoves.sort();
  
  if (onPC) { // move the principal continuation to the front of the list
    const Move& princCont = getPrincipalContinuation(depth);
    if (princCont.wasInvalid()) onPC = false; // don't use it
    else {
      listType::iterator listIterator=listOfMoves.begin();
      listType::iterator endOfList=listOfMoves.end();
      listType::iterator newEndOfList=listOfMoves.end();

      // remove the old entry
      newEndOfList=remove_if(listIterator, endOfList,
			     bind2nd(equal_to<pair<GeneratedMove,jint> >(), 
				     pair<GeneratedMove,jint>(GeneratedMove(princCont),0)));
      if (newEndOfList != endOfList) {
	listOfMoves.erase(newEndOfList,endOfList);
	// and put the new one at the front:
	listOfMoves.push_front(pair<GeneratedMove,jint>(princCont,12000));
      }
    }
  }

  if (extraDeep) {
    GeneratedMove gm2=GeneratedMove(gm);  // copy
    gm2.setEvaluateNow();                 // don't recurse further
    // form a binary tree of this position and it's descendent
    listOfMoves.push_front(pair<GeneratedMove,jint>(
			    GeneratedMove(gm2),gain));
  }
  
  listType::iterator listIterator=listOfMoves.begin();
  listType::iterator endOfList=listOfMoves.end();

  if (extraDeepPending) extraDeep=true; // now is the time to do this

  // in case of no alpha or beta cutoff
  if ( !onPC ) updatePrincipalContinuation(depth,listIterator->first.getMove()); 

  while (listIterator != endOfList) {

    GeneratedMove gm2=listIterator->first;
    jint thisGain=listIterator->second;
    jint value= -search(depth+1, thisMaxDepth, extraDeep, extraDeepPending, -beta, -alpha,
			gm2, thisGain, newHashcode, newChecksum, onPC);

    onPC = false;  // it can't be true anymore
    if (timeExhausted) {
      if (log && trace) {
	logFile << "<value>" << "time up" << "</value>" << endl;
	logFile << "</trace>" << endl;
      }
      return 0; // doesn't matter what we return
    }

    if (value == -ILLEGAL) { // illegal move
      listIterator=listOfMoves.erase(listIterator);
      repetitionCount+=1;
      continue;
    }

    if (value >= beta ) {
      if (!matingMode) store(hashIndex, newHashcode, newChecksum, value, maximumDepth-depth, 
	    LOWERBOUND, gm2.getMove());
      if (log && trace) {
	char buffer[30];
	principalContinuation[depth][depth].notate(buffer);
	logFile << "<principalContinuation bound=\"lower\" value=\"" << (even ? value : -value)
		<< "\">" << buffer <<  "</principalContinuation>" << endl;
	  } 
      if (depth > 0) gm.restorePosition(*homeBoard, checkRepetitions);
      // add the move to the killer list
      if (!matingMode) {
	killerList[depth].push_front(gm2.getMove());
	// keep the killer list short
	if (killerList[depth].size() > KILLER_LIST_SIZE) killerList[depth].pop_back(); 
      }
      if (log && trace) {
	logFile << "<value>" << value << "</value>" << endl;
	logFile << "</trace>" << endl;
      }
      if (log && matingAnalysis) {
        if (depth == 0) ;
	else if (even) logFile << "</evade>" << endl;
	else logFile << "</check>" << endl;
      }
      return value;
    }
    
    if (value > alpha) {
      alpha=value;
      exactMatch=true;
      updatePrincipalContinuation(depth,gm2.getMove());
      if (log && alphaCutoffs) {
	char buffer[30];
	gm2.getMove().notate(buffer);
	logFile << "<alphaCutoff bound=\"exact\" value=\"" << (even ? value : -value) 
		<< "\">" << buffer << "</alphaCutoff>" << endl;
      }
    }
    listIterator++;
  }
  if (depth > 0) gm.restorePosition(*homeBoard, checkRepetitions);

  if (!matingMode) store(hashIndex, newHashcode, newChecksum, alpha, maximumDepth-depth, 
	(exactMatch ? EXACT : UPPERBOUND), principalContinuation[depth][depth]);
  if (log && trace) {
    char buffer[30];
    principalContinuation[depth][depth].notate(buffer); 
    logFile << "<principalContinuation bound=\"" << (exactMatch ? "exact" : "upper") 
	    << "\" value=\""
	    << (even ? alpha : -alpha) << "\" black=\""
	    << principalContinuation[depth][depth].isBlack()
	    << "\">" << buffer <<  "</principalContinuation>" << endl;
  }
  if (log && trace) {
    logFile << "<value>" << alpha << "</value>" << endl;
    logFile << "</trace>" << endl;
  }
  if (log && matingAnalysis) {
    if (depth == 0) ;
    else if (even) logFile << "</evade>" << endl;
    else logFile << "</check>" << endl;
  }
  return alpha;   
}

jlong hash(const Board &bd)
{
  jlong result=0;

  for (int file=0; file < 12; file++)
    for (int rank=0; rank < 12; rank++) {
      int square=file*12 + rank;
      Cell &thisCell=bd.getCell(file,rank);
      Piece *piece=thisCell.getPiece();

      if (piece) {
	int colour=piece->isBlack();
	int delayed=piece->isDelayed();
	int ok=piece->isOK();
	int abbrev=piece->getAbbrev()-1;
	int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;
	result ^= pieceSquareTable[square][pieceIndex];
      }
    }
  return result;
}

// this should be called immediately prior to updatePosition()/performMove()
jlong adjustHash(jlong h, const Board &bd, const Move &m)
{

  int squares=m.squares();

  if (squares == 2 && m.getSource() == m.getStage2()) {
    if (!m.wasFirstCaptured()) return h; // pass move
    else {
      int X=m.getStage1().x();
      int Y=m.getStage1().y();
      int square=X*12 + Y;
      Piece *piece=bd.getCell(X,Y).getPiece(); // can't be null
      int colour=piece->isBlack();
      int delayed=piece->isDelayed();
      int ok=piece->isOK();
      int abbrev=piece->getAbbrev()-1;
      int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;

      return h^pieceSquareTable[square][pieceIndex]; // igui - just remove the piece
    }
  }

  // normal single or double move

  int targetX, targetY, target2X, target2Y, targetSquare, target2Square;
  
  jlong result=h;

  // remove the source piece
  int sourceX=m.getSource().x();
  int sourceY=m.getSource().y();
  int sourceSquare=sourceX*12 + sourceY;
  Piece *sourcePiece=bd.getCell(sourceX,sourceY).getPiece(); // can't be null
  int sourceColour=sourcePiece->isBlack();
  int sourceDelayed=sourcePiece->isDelayed();
  int sourceOk=sourcePiece->isOK();
  int sourceAbbrev=sourcePiece->getAbbrev()-1;
  int sourcePieceIndex=sourceAbbrev*8 + 4*sourceColour + 2*sourceDelayed + sourceOk;
  
  result ^= pieceSquareTable[sourceSquare][sourcePieceIndex];

  targetX=m.getStage1().x();
  targetY=m.getStage1().y();
  targetSquare=targetX*12 + targetY;
  
  if (m.wasFirstCaptured()) { // remove the target piece
    Piece *piece=bd.getCell(targetX,targetY).getPiece(); // can't be null
    int colour=piece->isBlack();
    int delayed=piece->isDelayed();
    int ok=piece->isOK();
    int abbrev=piece->getAbbrev()-1;
    int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;

    result ^= pieceSquareTable[targetSquare][pieceIndex];
  }

  if (squares==2) { // double move

    target2X=m.getStage2().x();
    target2Y=m.getStage2().y();
    target2Square=target2X*12 + target2Y;

    if (m.wasSecondCaptured()) { // remove the target2 piece
      Piece *piece=bd.getCell(target2X,target2Y).getPiece(); // can't be null
      int colour=piece->isBlack();
      int delayed=piece->isDelayed();
      int ok=piece->isOK();
      int abbrev=piece->getAbbrev()-1;
      int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;

      result ^= pieceSquareTable[target2Square][pieceIndex];
    }
    
    result ^= pieceSquareTable[target2Square][sourcePieceIndex];
    return result;
  }

  // single move

  result ^= pieceSquareTable[targetSquare][sourcePieceIndex];
  return result;
}


jint checksum(const Board &bd)
{
  jint result=0;

  for (int rank=0; rank < 12; rank++)
    for (int file=0; file < 12; file++) {
      int square=file*12 + rank;
      Cell &thisCell=bd.getCell(file,rank);
      Piece *piece=thisCell.getPiece();

      if (piece) {
	int colour=piece->isBlack();
	int delayed=piece->isDelayed();
	int ok=piece->isOK();
	int abbrev=piece->getAbbrev()-1;
	int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;
	result ^= checksumPieceSquareTable[square][pieceIndex];
      }
    }
  return result;
}

// this should be called immediately prior to updatePosition()/performMove()
jint adjustChecksum(jint ck, const Board &bd, const Move &m)
{

  int squares=m.squares();

  if (squares == 2 && m.getSource() == m.getStage2()) {
    if (!m.wasFirstCaptured()) return ck; // pass move
    else {
      int X=m.getStage1().x();
      int Y=m.getStage1().y();
      int square=X*12 + Y;
      Piece *piece=bd.getCell(X,Y).getPiece(); // can't be null
      int colour=piece->isBlack();
      int delayed=piece->isDelayed();
      int ok=piece->isOK();
      int abbrev=piece->getAbbrev()-1;
      int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;

      return ck^checksumPieceSquareTable[square][pieceIndex]; // igui - just remove the piece
    }
  }

  // normal single or double move

  int targetX, targetY, target2X, target2Y, targetSquare, target2Square;
  
  jlong result=ck;

  // remove the source piece
  int sourceX=m.getSource().x();
  int sourceY=m.getSource().y();
  int sourceSquare=sourceX*12 + sourceY;
  Piece *sourcePiece=bd.getCell(sourceX,sourceY).getPiece(); // can't be null
  int sourceColour=sourcePiece->isBlack();
  int sourceDelayed=sourcePiece->isDelayed();
  int sourceOk=sourcePiece->isOK();
  int sourceAbbrev=sourcePiece->getAbbrev()-1;
  int sourcePieceIndex=sourceAbbrev*8 + 4*sourceColour + 2*sourceDelayed + sourceOk;
  
  result ^= checksumPieceSquareTable[sourceSquare][sourcePieceIndex];

  targetX=m.getStage1().x();
  targetY=m.getStage1().y();
  targetSquare=targetX*12 + targetY;
  
  if (m.wasFirstCaptured()) { // remove the target piece
    Piece *piece=bd.getCell(targetX,targetY).getPiece(); // can't be null
    int colour=piece->isBlack();
    int delayed=piece->isDelayed();
    int ok=piece->isOK();
    int abbrev=piece->getAbbrev()-1;
    int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;

    result ^= checksumPieceSquareTable[targetSquare][pieceIndex];
  }

  if (squares==2) { // double move

    target2X=m.getStage2().x();
    target2Y=m.getStage2().y();
    target2Square=target2X*12 + target2Y;

    if (m.wasSecondCaptured()) { // remove the target2 piece
      Piece *piece=bd.getCell(target2X,target2Y).getPiece(); // can't be null
      int colour=piece->isBlack();
      int delayed=piece->isDelayed();
      int ok=piece->isOK();
      int abbrev=piece->getAbbrev()-1;
      int pieceIndex=abbrev*8 + 4*colour + 2*delayed + ok;

      result ^= checksumPieceSquareTable[target2Square][pieceIndex];
    }
    
    result ^= checksumPieceSquareTable[target2Square][sourcePieceIndex];
    return result;
  }

  // single move

  result ^= checksumPieceSquareTable[targetSquare][sourcePieceIndex];
  return result;
}

INLINE jint randomNumber(jint n)
{
  // if n > 0 return a pseudo-random number between 0 (inclusive) and n (exclusive).

  // if (n<=0) well, don't waste time checking for this

  if ((n & -n) == n) // i.e., n is a power of 2
    return jint(n * jlong(rand()) >> 31);

  jint bits, val;
  do {
    bits = rand();
    val = bits % n;
  } while(bits - val + (n-1) < 0);
  return val;
}

INLINE void Search::store(jint hashIndex, jlong newHashcode, jint newChecksum, jint value, 
			  jint depth, BoundType bound, const Move& theMove)
{
  

  jint multiplier=1;
  bool emptySlot=false;
  while (!emptySlot && multiplier <= hashMultiplier) { // looking for available slot
    HashEntry hashEntry=hashTable[hashIndex*multiplier];
    bool hashcodeOK=(hashEntry.getHashcode() == newHashcode);

    if (hashEntry.getHashcode() == 0 ) {                          // slot has never been used
      hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, 
						depth, bound, theMove);
      if (!hashcodeOK) cached+=1; // unused slot filled - permanent cache count usage has increased
      if (log && stores) logStore(bound, value, theMove, false);
      emptySlot=true;
    }
    else if (hashcodeOK && hashEntry.getChecksum() == newChecksum) { // replace this slot ?
      emptySlot=true;
      BoundType oldBound = hashEntry.getBound();
      jint oldValue = hashEntry.getValue();
      bool replace = false;

      switch (oldBound) {
      case EXACT: break; // don't bother replacing 
      case LOWERBOUND:   // beta cut off
	switch (bound) {
	case EXACT: // replace
	  hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, 
						    depth, bound, theMove);
	  replace = true; break;
	case LOWERBOUND: if (value > oldValue) {
	  hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, 
						    depth, bound, theMove);
	  replace = true; break;
	}
	case UPPERBOUND: if ( value == oldValue ) { // what else can it be?
	  hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, 
						    depth, EXACT, theMove);
	  replace = true; break;
	}
	}
      case UPPERBOUND:   // alpha cut off
	switch (bound) {
	case EXACT: // replace
	  hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, 
						    depth, bound, theMove);
	  replace = true; break;
	case LOWERBOUND: if (value == oldValue) { // what else can it be?
	  hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, 
						    depth, EXACT, theMove);
	  replace = true; break;
	}
	case UPPERBOUND: if ( value < oldValue ) { 
	  hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, 
						    depth, bound, theMove);
	  replace = true; break;
	}
	}
      }
      if (replace && log && stores) logStore(bound, value, theMove, true);
    }
    else if (hashcodeOK){         // therefore the checksums differ
      hashErrors+=1; // for statistics - checksums differ
      multiplier+=1; // look for next slot
    }
    else {                        // collisions - hashCodes differ
      collisions+=1; // for statistics - hashcodes differ
      multiplier+=1; // look for next slot
    }

  if (!emptySlot) { // pick a random slot for replacement (frequency counts will be better)
    replacements+=1;
    // we want a random number in the range (1,hashMultiplier)
    multiplier=randomNumber(hashMultiplier);
    multiplier+=1;
    hashTable[hashIndex*multiplier]=HashEntry(newHashcode, newChecksum, value, depth, 
					      bound, theMove);
    if (log && stores) logStore(bound, value, theMove, true);
  }
  }
}



INLINE void Search::logStore(BoundType bound, jint value, const Move &theMove, bool replaced)
{
  if (replaced) logFile << "<replace ";
  else logFile << "<store ";
  logFile << "value=\"" << value << "\" bound=\"" ;
  switch (bound) {
  case EXACT: logFile << "exact"; break;
  case UPPERBOUND: logFile << "upper"; break;
  case LOWERBOUND: logFile << "lower"; break;
  }
  logFile << "\" >";
  char buffer[30];
  theMove.notate(buffer);
  if (replaced) logFile << buffer << "</replace>" << endl;
  else logFile << buffer << "</store>" << endl;
}











