/**
 * player.cpp interface between Java and the C++ computer player
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

using namespace std;

#ifdef INLINE_OK 
#define INLINE inline
#else
#define INLINE  
#endif
#include "Player.jnih"
#include "piece.h"
#include "board.h"
#include "coordinates.h"
#include "move.h"
#include "search.h"
#include <iostream>
#include <fstream>
#include <strstream.h>
#include <ctime>
#include <csignal>
#include <unistd.h>                       // this means a POSIX-compliant O/S is needed

jlong        pieceSquareTable[144][312];  // for calculating the hashcode
jint checksumPieceSquareTable[144][312];  // for calculating the checksum
jint                        hashEntries;  // how many entries in the hashtable
jshort                   hashMultiplier;  // how many duplicate slots for a given hashcode
jlong       hashBits=0x000000000000ffff;  // bit mask of the hashcode, to use as an index 
jint Search::cached;
bool timeExhausted;
  
HashEntry        *hashTable=NULL;  // address of the hashtable

static jint count;
static jint exactHits;
static jint upperCutoffs;
static jint lowerCutoffs;
static jint nearMisses;
static jint hashingCollisions;
static jint hashingErrors;
static jint replacements;
static jint repetitionCount;
static Move bestMove;

ofstream           statsFile;

void catch_alarm (int);            // forward declaration
void getStats(Search&);            // forward declaration

JNIEXPORT jobject JNICALL Java_Player_runComputer(JNIEnv *env, jobject obj)
{
  jclass         playerClass;             // class ID of Player
  jfieldID         colourID;
  jboolean           colour;              // the computer-player's colour
  jfieldID  alwaysPromoteID;
  jboolean    alwaysPromote;              
  jfieldID   previousMoveID;
  jobject      previousMove;
  jfieldID      thisBoardID;
  jobject         thisBoard;
  jfieldID          lionsID;
  jint                lions;
  jfieldID         initTTID;
  jboolean           initTT;
  jfieldID           timeID;
  jfieldID        maxTimeID;
  jlong           basicTime;
  jint          maximumTime;


  // first, get the class and fields of Player

  playerClass=env->GetObjectClass(obj);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get class of Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  initTTID=env->GetStaticFieldID(playerClass,"buildTranspositionTable","Z");
  if (initTTID == NULL) {
    cerr << "Unable to static find field: 'buildTranspositionTable' in Player" << endl;
    return NULL;
  }
  initTT=env->GetStaticBooleanField(playerClass,initTTID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get buildTranspositionTable in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }


  // now find out if the user wants stats logging

  jfieldID  logStatsID;

  logStatsID=env->GetFieldID(playerClass,"logStats","Z");
  if (logStatsID == NULL) {
    cerr << "Unable to find field: 'logStats' in Player" << endl;
    return NULL;
  }

  jboolean logStats=env->GetBooleanField(obj,logStatsID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get logStats in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  // if buildTranspositionTable is true, then we need to build the piece table
  // and the transposition table

  if (initTT) {
    env->SetStaticBooleanField(playerClass,initTTID,false); // don't do this again

  if (logStats) statsFile.open("stats.log", ios::out | ios::app | ios::trunc);

    Search::cached=0;
 
    jfieldID            generatorID;
    jobject   randomNumberGenerator;
    jmethodID            nextLongID;
    jmethodID                nextID;

    generatorID=env->GetFieldID(playerClass,"randomNumberGenerator","Ljava/util/Random;");
    if (generatorID == NULL) {
      cerr << "Unable to find field: 'randomNumberGenerator' in Player" << endl;
      return NULL;
    }

    randomNumberGenerator=env->GetObjectField(obj,generatorID);
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to get randomNumberGenerator in Player" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }

    jclass randomClass=env->FindClass("java/util/Random");
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to find class Random" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }

    nextLongID=env->GetMethodID(randomClass,"nextLong","()J");
    if (nextLongID == NULL) {
      cerr << "Unable to find method: 'nextLong()' in java/util/Random" << endl;
      return NULL;
    }


    nextID=env->GetMethodID(randomClass,"nextInt","()I");
    if (nextID == NULL) {
      cerr << "Unable to find method: 'nextInt()' in java/util/Random" << endl;
      return NULL;
    }

    // now initialise the piece-square table with random 64-bit numbers

    for (int file=0; file < 12; file++)
      for (int rank=0; rank < 12; rank++) {
	int square=file*12 + rank;
	for (int piece=0; piece < int(LION); piece++)
	  for (int extra=0; extra < 8; extra++) { 
	    int pieceIndex=piece*8 + extra;
	    pieceSquareTable[square][pieceIndex]=
	      env->CallLongMethod(randomNumberGenerator,nextLongID);
	    if (env->ExceptionOccurred()) {
	      cerr << "Exception Occurred trying to get a random long number" << endl;
	      env->ExceptionDescribe();
	      env->ExceptionClear();
	      return NULL;
	    }
	  }
      }

    // now initialise the checksum piece-square table with random 32-bit numbers

    for (int file=0; file < 12; file++)
      for (int rank=0; rank < 12; rank++) {
	int square=file*12 + rank;
	for (int piece=0; piece < int(LION); piece++)
	  for (int extra=0; extra < 8; extra++) { 
	    int pieceIndex=piece*8 + extra;
	    checksumPieceSquareTable[square][pieceIndex]=
	      env->CallIntMethod(randomNumberGenerator,nextID);
	    if (env->ExceptionOccurred()) {
	      cerr << "Exception Occurred trying to get a random number" << endl;
	      env->ExceptionDescribe();
	      env->ExceptionClear();
	      return NULL;
	    }
	  }
      }

    // now build the transition table itself

    // each entry is 24(?) bytes
    
    jfieldID  mbID, mrID;

    mbID=env->GetFieldID(playerClass,"mb","I");
    if (mbID == NULL) {
      cerr << "Unable to find field: 'mb' in Player" << endl;
      return NULL;
    }

    mrID=env->GetFieldID(playerClass,"mr","I");
    if (mrID == NULL) {
      cerr << "Unable to find field: 'mr' in Player" << endl;
      return NULL;
    }

    hashMultiplier=env->GetIntField(obj,mrID);
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to get mr in Player" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }    

    jint hashSizeFactor=env->GetIntField(obj,mbID);
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to get mb in Player" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }    
  
    hashBits=0x000000000000ffff  | ((hashSizeFactor-1) << 16);
    hashEntries=hashMultiplier*hashSizeFactor*64*1024; // multiples of 1.5 MB 
    
    if (hashTable) delete[] hashTable;
    hashTable=new HashEntry[hashEntries];
  }

  colourID=env->GetFieldID(playerClass,"colour","Z");
  if (colourID == NULL) {
    cerr << "Unable to find field: 'colour' in Player" << endl;
    return NULL;
  }
  colour=env->GetBooleanField(obj,colourID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get colour in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  lionsID=env->GetFieldID(playerClass,"lions","I");
  if (lionsID == NULL) {
    cerr << "Unable to find field: 'lions' in Player" << endl;
    return NULL;
  }
  lions=env->GetIntField(obj,lionsID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get lions in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  timeID=env->GetFieldID(playerClass,"basicTime","J");
  if (timeID == NULL) {
    cerr << "Unable to find field: 'basicTime' in Player" << endl;
    return NULL;
  }
  basicTime=env->GetLongField(obj,timeID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get basicTime in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  maxTimeID=env->GetFieldID(playerClass,"maximumTime","I");
  if (maxTimeID == NULL) {
    cerr << "Unable to find field: 'maximumTime' in Player" << endl;
    return NULL;
  }
  maximumTime=env->GetLongField(obj,maxTimeID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get maximumTime in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  alwaysPromoteID=env->GetFieldID(playerClass,"alwaysPromote","Z");
  if (alwaysPromoteID == NULL) {
    cerr << "Unable to find field: 'alwaysPromote' in Player" << endl;
    return NULL;
  }
  alwaysPromote=env->GetBooleanField(obj,alwaysPromoteID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get alwaysPromote in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  previousMoveID=env->GetFieldID(playerClass,"previousMove","LMove;");
  if (previousMoveID == NULL) {
    cerr << "Unable to find field: 'previousMove' in Player" << endl;
    return NULL;
  }
  previousMove=env->GetObjectField(obj,previousMoveID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get previousMove in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  thisBoardID=env->GetFieldID(playerClass,"thisBoard","LBoard;");
  if (thisBoardID == NULL) {
    cerr << "Unable to find field: 'thisBoard' in Player" << endl;
    return NULL;
  }
  thisBoard=env->GetObjectField(obj,thisBoardID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get thisBoard in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  // now, get the class and fields of previousMove

  jclass                moveClass;
  jfieldID               numberID;
  jint                     number;
  jfieldID               abbrevID;
  jstring                  abbrev;
  jfieldID               sourceID;
  jobject                  source;
  jfieldID          destinationID;
  jobject             destination;
  jfieldID         destination2ID;
  jobject            destination2;
  jfieldID   destinationSquaresID;
  jbyte        destinationSquares;     
  jfieldID         firstCaptureID;
  jboolean           firstCapture;     
  jfieldID        secondCaptureID;
  jboolean          secondCapture;     
  jfieldID            promotionID;
  jboolean              promotion;     
  jfieldID              mcolourID;
  jboolean                mcolour;  // move colour     
  jfieldID           lionSourceID;
  jboolean             lionSource;     
  jfieldID           lionTargetID;
  jboolean             lionTarget;     
  jfieldID  anotherMovePossibleID;
  jboolean    anotherMovePossible;
  jfieldID        doubleCaptureID;
  jboolean          doubleCapture;
  jfieldID         lionCapturedID;
  jboolean           lionCaptured;
  jfieldID        didNotPromoteID;
  jboolean          didNotPromote;
  jfieldID          promotionOKID;
  jboolean            promotionOK;
  jfieldID              delayedID;
  jboolean                delayed;
     
  moveClass=env->FindClass("Move");
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get class of Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  numberID=env->GetFieldID(moveClass,"number","I");
  if (numberID == NULL) {
    cerr << "Unable to find field: 'number' in Move" << endl;
    return NULL;
  }
  number=env->GetIntField(previousMove,numberID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get number in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  abbrevID=env->GetFieldID(moveClass,"abbrev","Ljava/lang/String;");
  if (abbrevID == NULL) {
    cerr << "Unable to find field: 'abbrev' in Move" << endl;
    return NULL;
  }
  abbrev=static_cast<jstring>(env->GetObjectField(previousMove,abbrevID));
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get abbrev in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  sourceID=env->GetFieldID(moveClass,"source","LCoordinates;");
  if (sourceID == NULL) {
    cerr << "Unable to find field: 'source' in Move" << endl;
    return NULL;
  }
  source=env->GetObjectField(previousMove,sourceID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get source in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  destinationID=env->GetFieldID(moveClass,"destination","LCoordinates;");
  if (destinationID == NULL) {
    cerr << "Unable to find field: 'destination' in Move" << endl;
    return NULL;
  }
  destination=env->GetObjectField(previousMove,destinationID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get destination in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  destination2ID=env->GetFieldID(moveClass,"destination2","LCoordinates;");
  if (destination2ID == NULL) {
    cerr << "Unable to find field: 'destination2' in Move" << endl;
    return NULL;
  }
  destination2=env->GetObjectField(previousMove,destination2ID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get destination2 in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  destinationSquaresID=env->GetFieldID(moveClass,"destinationSquares","B");
  if (destinationSquaresID == NULL) {
    cerr << "Unable to find field: 'destinationSquares' in Move" << endl;
    return NULL;
  }
  destinationSquares=env->GetByteField(previousMove,destinationSquaresID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get destinationSquares in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  firstCaptureID=env->GetFieldID(moveClass,"firstCapture","Z");
  if (firstCaptureID == NULL) {
    cerr << "Unable to find field: 'firstCapture' in Move" << endl;
    return NULL;
  }
  firstCapture=env->GetBooleanField(previousMove,firstCaptureID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get firstCapture in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  secondCaptureID=env->GetFieldID(moveClass,"secondCapture","Z");
  if (secondCaptureID == NULL) {
    cerr << "Unable to find field: 'secondCapture' in Move" << endl;
    return NULL;
  }
  secondCapture=env->GetBooleanField(previousMove,secondCaptureID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get secondCapture in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  promotionID=env->GetFieldID(moveClass,"promotion","Z");
  if (promotionID == NULL) {
    cerr << "Unable to find field: 'promotion' in Move" << endl;
    return NULL;
  }
  promotion=env->GetBooleanField(previousMove,promotionID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get promotion in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  mcolourID=env->GetFieldID(moveClass,"colour","Z");
  if (mcolourID == NULL) {
    cerr << "Unable to find field: 'colour' in Move" << endl;
    return NULL;
  }
  mcolour=env->GetBooleanField(previousMove,mcolourID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get colour in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  lionSourceID=env->GetFieldID(moveClass,"lionSource","Z");
  if (lionSourceID == NULL) {
    cerr << "Unable to find field: 'lionSource' in Move" << endl;
    return NULL;
  }
  lionSource=env->GetBooleanField(previousMove,lionSourceID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get lionSource in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  lionTargetID=env->GetFieldID(moveClass,"lionTarget","Z");
  if (lionTargetID == NULL) {
    cerr << "Unable to find field: 'lionTarget' in Move" << endl;
    return NULL;
  }
  lionTarget=env->GetBooleanField(previousMove,lionTargetID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get colour in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  anotherMovePossibleID=env->GetFieldID(moveClass,"anotherMovePossible","Z");
  if (anotherMovePossibleID == NULL) {
    cerr << "Unable to find field: 'anotherMovePossible' in Move" << endl;
    return NULL;
  }
  anotherMovePossible=env->GetBooleanField(previousMove,anotherMovePossibleID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get anotherMovePossible in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  doubleCaptureID=env->GetFieldID(moveClass,"doubleCapture","Z");
  if (doubleCaptureID == NULL) {
    cerr << "Unable to find field: 'doubleCapture' in Move" << endl;
    return NULL;
  }
  doubleCapture=env->GetBooleanField(previousMove,doubleCaptureID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get doubleCapture in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  lionCapturedID=env->GetFieldID(moveClass,"lionCaptured","Z");
  if (lionCapturedID == NULL) {
    cerr << "Unable to find field: 'lionCaptured' in Move" << endl;
    return NULL;
  }
  lionCaptured=env->GetBooleanField(previousMove,lionCapturedID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get lionCaptured in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  didNotPromoteID=env->GetFieldID(moveClass,"didNotPromote","Z");
  if (didNotPromoteID == NULL) {
    cerr << "Unable to find field: 'didNotPromote' in Move" << endl;
    return NULL;
  }
  didNotPromote=env->GetBooleanField(previousMove,didNotPromoteID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get didNotPromote in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  promotionOKID=env->GetFieldID(moveClass,"promotionOK","Z");
  if (promotionOKID == NULL) {
    cerr << "Unable to find field: 'promotionOK' in Move" << endl;
    return NULL;
  }
  promotionOK=env->GetBooleanField(previousMove,promotionOKID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get promotionOK in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  delayedID=env->GetFieldID(moveClass,"delayedPromotion","Z");
  if (delayedID == NULL) {
    cerr << "Unable to find field: 'delayedPromotion' in Move" << endl;
    return NULL;
  }
  delayed=env->GetBooleanField(previousMove,delayedID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get delayed in Move" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  // next, translate the abbrev string to a PieceType

  const char *abbrevStr=env->GetStringUTFChars(abbrev, 0);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to GetStringUTFChars for abbrev" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  PieceType moveAbbrev=NONE;
  if (strcmp(abbrevStr,"P") == 0) moveAbbrev=PAWN;
  else if (strcmp(abbrevStr,"+P") == 0) moveAbbrev=TOKIN;
  else if (strcmp(abbrevStr,"GB") == 0) moveAbbrev=GOBETWEEN;
  else if (strcmp(abbrevStr,"+GB") == 0) moveAbbrev=PrGOBETWEEN;
  else if (strcmp(abbrevStr,"C") == 0) moveAbbrev=COPPER;
  else if (strcmp(abbrevStr,"+C") == 0) moveAbbrev=PrCOPPER;
  else if (strcmp(abbrevStr,"S") == 0) moveAbbrev=SILVER;
  else if (strcmp(abbrevStr,"+S") == 0) moveAbbrev=PrSILVER;
  else if (strcmp(abbrevStr,"G") == 0) moveAbbrev=GOLD;
  else if (strcmp(abbrevStr,"+G") == 0) moveAbbrev=PrGOLD;
  else if (strcmp(abbrevStr,"FL") == 0) moveAbbrev=LEOPARD;
  else if (strcmp(abbrevStr,"+FL") == 0) moveAbbrev=PrLEOPARD;
  else if (strcmp(abbrevStr,"DE") == 0) moveAbbrev=DRUNKELEPHANT;
  else if (strcmp(abbrevStr,"+DE") == 0) moveAbbrev=PrDRUNKELEPHANT;
  else if (strcmp(abbrevStr,"K") == 0) moveAbbrev=KING;
  else if (strcmp(abbrevStr,"L") == 0) moveAbbrev=LANCE;
  else if (strcmp(abbrevStr,"+L") == 0) moveAbbrev=WHITEHORSE;
  else if (strcmp(abbrevStr,"RC") == 0) moveAbbrev=REVERSECHARIOT;
  else if (strcmp(abbrevStr,"+RC") == 0) moveAbbrev=WHALE;
  else if (strcmp(abbrevStr,"BT") == 0) moveAbbrev=BLINDTIGER;
  else if (strcmp(abbrevStr,"+BT") == 0) moveAbbrev=FLYINGSTAG;
  else if (strcmp(abbrevStr,"Ky") == 0) moveAbbrev=KYLIN;
  else if (strcmp(abbrevStr,"+Ky") == 0) moveAbbrev=PrKYLIN;
  else if (strcmp(abbrevStr,"Ph") == 0) moveAbbrev=PHOENIX;
  else if (strcmp(abbrevStr,"+Ph") == 0) moveAbbrev=PrPHOENIX;
  else if (strcmp(abbrevStr,"B") == 0) moveAbbrev=BISHOP;
  else if (strcmp(abbrevStr,"+B") == 0) moveAbbrev=PrBISHOP;
  else if (strcmp(abbrevStr,"SM") == 0) moveAbbrev=SIDEMOVER;
  else if (strcmp(abbrevStr,"+SM") == 0) moveAbbrev=FREEBOAR;
  else if (strcmp(abbrevStr,"VM") == 0) moveAbbrev=VERTICALMOVER;
  else if (strcmp(abbrevStr,"+VM") == 0) moveAbbrev=FLYINGOX;
  else if (strcmp(abbrevStr,"R") == 0) moveAbbrev=ROOK;
  else if (strcmp(abbrevStr,"+R") == 0) moveAbbrev=PrROOK;
  else if (strcmp(abbrevStr,"DH") == 0) moveAbbrev=HORSE;
  else if (strcmp(abbrevStr,"+DH") == 0) moveAbbrev=FALCON;
  else if (strcmp(abbrevStr,"DK") == 0) moveAbbrev=DRAGON;
  else if (strcmp(abbrevStr,"+DK") == 0) moveAbbrev=EAGLE;
  else if (strcmp(abbrevStr,"FK") == 0) moveAbbrev=FREEKING;
  else if (strcmp(abbrevStr,"Ln") == 0) moveAbbrev=LION;
  else cerr << "Unknown piece type in previousMove: " << abbrevStr << "###" << endl;

  env->ReleaseStringUTFChars(abbrev, abbrevStr);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to ReleaseStringUTFChars for abbrev" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  // next, we need to build three Coordinates structures

  jclass   coordinatesClass;
  jfieldID              XID;
  jfieldID              YID;
  jint              sourceX;
  jint              sourceY;
  jint         destinationX;
  jint         destinationY;
  jint        destination2X;
  jint        destination2Y;

  coordinatesClass=env->FindClass("Coordinates");
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to find class for Coordinates" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  XID=env->GetFieldID(coordinatesClass,"X","I");
  if (XID == NULL) {
    cerr << "Unable to find field: 'X' in Coordinates" << endl;
    return NULL;
  }
  YID=env->GetFieldID(coordinatesClass,"Y","I");
  if (YID == NULL) {
    cerr << "Unable to find field: 'Y' in Coordinates" << endl;
    return NULL;
  }
  sourceX=env->GetIntField(source,XID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get sourceX in Coordinates" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  sourceY=env->GetIntField(source,YID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get sourceY in Coordinates" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  const Coordinates sourceC(sourceX,sourceY);

  destinationX=env->GetIntField(destination,XID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get destinationX in Coordinates" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  destinationY=env->GetIntField(destination,YID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get destinationY in Coordinates" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  const Coordinates destinationC(destinationX,destinationY);
  
  destination2X=env->GetIntField(destination2,XID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get destination2X in Coordinates" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  destination2Y=env->GetIntField(destination2,YID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get destination2Y in Coordinates" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  const Coordinates destination2C(destination2X,destination2Y);

  const Move humanMove(number,moveAbbrev,sourceC,destinationC,destination2C,destinationSquares,
		       firstCapture,secondCapture,promotion,promotionOK,delayed,mcolour,
		       lionSource,anotherMovePossible,doubleCapture,lionCaptured,lionTarget,
		       didNotPromote);

  // now, get the class and fields of thisBoard

  jclass               boardClass;
  jfieldID        piecesOnBoardID;
  jbyteArray        piecesOnBoard;
  jbyte            piecesOnBoard0;
  jbyte            piecesOnBoard1;
  jfieldID         crownPrincesID;
  jbyteArray         crownPrinces;
  jbyte             crownPrinces0;
  jbyte             crownPrinces1;
  jfieldID            positionsID;
  jobjectArray          positions;
  jobject              positions0;
  jobject              positions1;
      
  boardClass=env->FindClass("Board");
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to find class for Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  piecesOnBoardID=env->GetFieldID(boardClass,"piecesOnBoard","[B");
  if (piecesOnBoardID == NULL) {
    cerr << "Unable to find field: 'piecesOnBoard' in Board" << endl;
    return NULL;
  }
  piecesOnBoard=static_cast<jbyteArray>(env->GetObjectField(thisBoard,piecesOnBoardID));
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get piecesOnBoard in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  jbyte *pieces=env->GetByteArrayElements(piecesOnBoard, 0);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get piecesOnBoard[0,1] in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  piecesOnBoard0=pieces[0];
  piecesOnBoard1=pieces[1];
  env->ReleaseByteArrayElements(piecesOnBoard,pieces, 0);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to release piecesOnBoard in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  crownPrincesID=env->GetFieldID(boardClass,"crownPrinces","[B");
  if (crownPrincesID == NULL) {
    cerr << "Unable to find field: 'crownPrinces' in Board" << endl;
    return NULL;
  }
  crownPrinces=static_cast<jbyteArray>(env->GetObjectField(thisBoard,crownPrincesID));
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get crownPrinces in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  } 
  jbyte *princes=env->GetByteArrayElements(crownPrinces, 0);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get crownPrinces[0,1] in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  crownPrinces0=princes[0];
  crownPrinces1=princes[1];
  env->ReleaseByteArrayElements(crownPrinces, princes, 0);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to release crownPrinces in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  positionsID=env->GetFieldID(boardClass,"positions","[Ljava/util/HashSet;");
  if (positionsID == NULL) {
    cerr << "Unable to find field: 'positions' in Board" << endl;
    return NULL;
  }
  positions=static_cast<jobjectArray>(env->GetObjectField(thisBoard,positionsID));  
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get positions in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  positions0=env->GetObjectArrayElement(positions, 0);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get positions[0] in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  positions1=env->GetObjectArrayElement(positions, 1);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get positions[1] in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  // now construct a board, and fill in the fields

  Board homeBoard(lions);

  // first the easy bits

  homeBoard.setPiecesOnBoard(jint(piecesOnBoard0),0);
  homeBoard.setPiecesOnBoard(jint(piecesOnBoard1),1);
  homeBoard.setCrownPrinces(jint(crownPrinces0),0);
  homeBoard.setCrownPrinces(jint(crownPrinces1),1);

  // next the positions sets

  jclass hashset=env->FindClass("java/util/HashSet");
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to find class HashSet" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  jclass iteratorClass=env->FindClass("java/util/Iterator");
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to find class Iterator" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  jmethodID iteratorID=env->GetMethodID(hashset,"iterator","()Ljava/util/Iterator;");
  if (iteratorID == NULL) {
    cerr << "Unable to find method: 'iterator()' in java/util/HashSet" << endl;
    return NULL;
  }
  jobject blackIterator=env->CallObjectMethod(positions1,iteratorID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get blackIterator" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  jobject whiteIterator=env->CallObjectMethod(positions0,iteratorID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get whiteIterator" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  jmethodID hasNextID=env->GetMethodID(iteratorClass,"hasNext","()Z");
  if (hasNextID == NULL) {
    cerr << "Unable to find method: 'hasNext()' in java/util/Iterator" << endl;
    return NULL;
  }
  jmethodID nextID=env->GetMethodID(iteratorClass,"next","()Ljava/lang/Object;");
  if (nextID == NULL) {
    cerr << "Unable to find method: 'next()' in java/util/Iterator" << endl;
    return NULL;
  }
  // now scan the black positions set

  while (env->CallBooleanMethod(blackIterator,hasNextID))
  {
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling blackIterator.hasNext()" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }

    jstring thisForsyth=static_cast<jstring>(env->CallObjectMethod(blackIterator,nextID));
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling blackIterator.next()" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }  

    const char *blackStr=env->GetStringUTFChars(thisForsyth, 0);
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling GetStringUTFChars for black Forsyth" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }
    string copiedForsyth(blackStr);
    env->ReleaseStringUTFChars(thisForsyth, blackStr);  
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling ReleaseStringUTFChars for black Forsyth" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }
    homeBoard.addPosition(copiedForsyth, true);
  }

  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred after while blackIterator.hasNext()" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  // now scan the white positions set

  while (env->CallBooleanMethod(whiteIterator,hasNextID)) {
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling whiteIterator.hasNext()" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }

    jstring thisForsyth=static_cast<jstring>(env->CallObjectMethod(whiteIterator,nextID));
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling blackIterator.next()" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }  
    const char *whiteStr=env->GetStringUTFChars(thisForsyth, 0);
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling GetStringUTFChars for white Forsyth" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }
    string copiedForsyth(whiteStr);
    env->ReleaseStringUTFChars(thisForsyth, whiteStr);  
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred calling ReleaseStringUTFChars for white Forsyth" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }
    homeBoard.addPosition(copiedForsyth, false);
  }

  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred after while whiteIterator.hasNext()" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  // now we must set up each cell

  jfieldID cellsID=env->GetFieldID(boardClass,"cells","[[LCell;");  
  if (cellsID == NULL) {
    cerr << "Unable to find field: 'cells' in Board" << endl;
    return NULL;
  }

  jobjectArray cells=static_cast<jobjectArray>(env->GetObjectField(thisBoard,cellsID));
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred getting cells in Board" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }
  bool first=true;
  for (jint file=0; file < 12; file++) {
    jobjectArray thisFile=static_cast<jobjectArray>(env->GetObjectArrayElement(cells, file));
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred getting thisFile in Board" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }
    
    for (jint rank=0; rank< 12; rank++) {
      jobject thisCell=env->GetObjectArrayElement(thisFile, rank);
      if (env->ExceptionOccurred()) {
	cerr << "Exception Occurred getting Cell in Board" << endl;
	env->ExceptionDescribe();
	env->ExceptionClear();
	return NULL;
      }
      Cell &cell=homeBoard.getCell(file,rank);
      // now we need to process the per-Cell information

      jclass cellClass=env->FindClass("Cell");      
      if (env->ExceptionOccurred()) {
	cerr << "Exception Occurred finding class Cell" << endl;
	env->ExceptionDescribe();
	env->ExceptionClear();
	return NULL;
      }
      jclass pieceClass=env->FindClass("PieceRepresentation");
      if (env->ExceptionOccurred()) {
	cerr << "Exception Occurred finding class PieceRepresentation" << endl;
	env->ExceptionDescribe();
	env->ExceptionClear();
	return NULL;
      }
      jfieldID lightingID=env->GetFieldID(cellClass,"lighting","I");        
      if (lightingID == NULL) {
	cerr << "Unable to find field: 'lighting' in Cell" << endl;
	return NULL;
      }
      jfieldID pieceID=env->GetFieldID(cellClass,"piece","LPieceRepresentation;");        
      if (pieceID == NULL) {
	cerr << "Unable to find field: 'piece' in Cell" << endl;
	return NULL;
      }

      jint light=env->GetIntField(thisCell,lightingID);
      if (env->ExceptionOccurred()) {
	cerr << "Exception Occurred tryng to get light in Cell" << endl;
	env->ExceptionDescribe();
	env->ExceptionClear();
	return NULL;
      }
      if (light != CLEAR) cell.setLighting(Lighting(light));

      jobject piece=env->GetObjectField(thisCell,pieceID);
      if (env->ExceptionOccurred()) {
	cerr << "Exception Occurred tryng to get piece in Cell" << endl;
	env->ExceptionDescribe();
	env->ExceptionClear();
	return NULL;
      }
      jboolean colour=false, promoted=false, delayed=false, OK=false;

      if (piece != NULL) { // we must set up the new piece)

	jmethodID abbrevID=env->GetMethodID(pieceClass,"getAbbrev","()Ljava/lang/String;");
	if (abbrevID == NULL) {
	  cerr << "Unable to find method: 'getAbbrev()' in PieceReprestnation" << endl;	
	  return NULL;
	}
	jfieldID colourID=env->GetFieldID(pieceClass,"colour","Z");        
	if (colourID == NULL) {
	  cerr << "Unable to find field: 'colour' in PieceReprestnation" << endl;	
	  return NULL;
	}
        else colour=env->GetBooleanField(piece,colourID);
	if (env->ExceptionOccurred()) {
	  cerr << "Exception Occurred tryng to get colour in PieceRepresentation" << endl;
	  env->ExceptionDescribe();
	  env->ExceptionClear();
	  return NULL;
	}
	jfieldID promotedID=env->GetFieldID(pieceClass,"promoted","Z");        
	if (promotedID == NULL) { 
	  cerr << "Unable to find field: 'promoted' in PieceReprestnation" << endl;	
	  return NULL;
	}
        else promoted=env->GetBooleanField(piece,promotedID);
	if (env->ExceptionOccurred()) {
	  cerr << "Exception Occurred tryng to get promoted in PieceRepresentation" << endl;
	  env->ExceptionDescribe();
	  env->ExceptionClear();
	  return NULL;
	}
	jfieldID delayedID=env->GetFieldID(pieceClass,"delayedPromotion","Z");        
	if (delayedID == NULL) {
	  cerr << "Unable to find field: 'delayedPromotion' in PieceReprestnation" << endl;	
	  return NULL;
	}
        else delayed=env->GetBooleanField(piece,delayedID);
	if (env->ExceptionOccurred()) {
	  cerr << "Exception Occurred tryng to get delayed in PieceRepresentation" << endl;
	  env->ExceptionDescribe();
	  env->ExceptionClear();
	  return NULL;
	}
	jfieldID okID=env->GetFieldID(pieceClass,"promotionOK","Z");        
	if (okID == NULL) {
	  cerr << "Unable to find field: 'promotionOK' in PieceReprestnation" << endl;	
	  return NULL;
	}
        else OK=env->GetBooleanField(piece,okID);
	if (env->ExceptionOccurred()) {
	  cerr << "Exception Occurred trying to get promotionOK in PieceRepresentation" << endl;
	  env->ExceptionDescribe();
	  env->ExceptionClear();
	  return NULL;
	}
	jstring abbrevString=static_cast<jstring>(env->CallObjectMethod(piece,abbrevID));
	if (env->ExceptionOccurred()) {
	  cerr << "Exception Occurred trying to get abbrevString in PieceRepresentation" << endl;
	  env->ExceptionDescribe();
	  env->ExceptionClear();
	  return NULL;
	}
	const char *abbrevStr=env->GetStringUTFChars(abbrevString, 0);
	if (env->ExceptionOccurred()) {
	  cerr << "Exception Occurred trying to GetStringUTFChars for abbrevStr" << endl;
	  env->ExceptionDescribe();
	  env->ExceptionClear();
	  return NULL;
	}
	PieceType moveAbbrev=NONE;
	if (strcmp(abbrevStr,"P") == 0) moveAbbrev=PAWN;
	else if (strcmp(abbrevStr,"+P") == 0) moveAbbrev=TOKIN;
	else if (strcmp(abbrevStr,"GB") == 0) moveAbbrev=GOBETWEEN;
	else if (strcmp(abbrevStr,"+GB") == 0) moveAbbrev=PrGOBETWEEN;
	else if (strcmp(abbrevStr,"C") == 0) moveAbbrev=COPPER;
	else if (strcmp(abbrevStr,"+C") == 0) moveAbbrev=PrCOPPER;
	else if (strcmp(abbrevStr,"S") == 0) moveAbbrev=SILVER;
	else if (strcmp(abbrevStr,"+S") == 0) moveAbbrev=PrSILVER;
	else if (strcmp(abbrevStr,"G") == 0) moveAbbrev=GOLD;
	else if (strcmp(abbrevStr,"+G") == 0) moveAbbrev=PrGOLD;
	else if (strcmp(abbrevStr,"FL") == 0) moveAbbrev=LEOPARD;
	else if (strcmp(abbrevStr,"+FL") == 0) moveAbbrev=PrLEOPARD;
	else if (strcmp(abbrevStr,"DE") == 0) moveAbbrev=DRUNKELEPHANT;
	else if (strcmp(abbrevStr,"+DE") == 0) moveAbbrev=PrDRUNKELEPHANT;
	else if (strcmp(abbrevStr,"K") == 0) moveAbbrev=KING;
	else if (strcmp(abbrevStr,"L") == 0) moveAbbrev=LANCE;
	else if (strcmp(abbrevStr,"+L") == 0) moveAbbrev=WHITEHORSE;
	else if (strcmp(abbrevStr,"RC") == 0) moveAbbrev=REVERSECHARIOT;
	else if (strcmp(abbrevStr,"+RC") == 0) moveAbbrev=WHALE;
	else if (strcmp(abbrevStr,"BT") == 0) moveAbbrev=BLINDTIGER;
	else if (strcmp(abbrevStr,"+BT") == 0) moveAbbrev=FLYINGSTAG;
	else if (strcmp(abbrevStr,"Ky") == 0) moveAbbrev=KYLIN;
	else if (strcmp(abbrevStr,"+Ky") == 0) moveAbbrev=PrKYLIN;
	else if (strcmp(abbrevStr,"Ph") == 0) moveAbbrev=PHOENIX;
	else if (strcmp(abbrevStr,"+Ph") == 0) moveAbbrev=PrPHOENIX;
	else if (strcmp(abbrevStr,"B") == 0) moveAbbrev=BISHOP;
	else if (strcmp(abbrevStr,"+B") == 0) moveAbbrev=PrBISHOP;
	else if (strcmp(abbrevStr,"SM") == 0) moveAbbrev=SIDEMOVER;
	else if (strcmp(abbrevStr,"+SM") == 0) moveAbbrev=FREEBOAR;
	else if (strcmp(abbrevStr,"VM") == 0) moveAbbrev=VERTICALMOVER;
	else if (strcmp(abbrevStr,"+VM") == 0) moveAbbrev=FLYINGOX;
	else if (strcmp(abbrevStr,"R") == 0) moveAbbrev=ROOK;
	else if (strcmp(abbrevStr,"+R") == 0) moveAbbrev=PrROOK;
	else if (strcmp(abbrevStr,"DH") == 0) moveAbbrev=HORSE;
	else if (strcmp(abbrevStr,"+DH") == 0) moveAbbrev=FALCON;
	else if (strcmp(abbrevStr,"DK") == 0) moveAbbrev=DRAGON;
	else if (strcmp(abbrevStr,"+DK") == 0) moveAbbrev=EAGLE;
	else if (strcmp(abbrevStr,"FK") == 0) moveAbbrev=FREEKING;
	else if (strcmp(abbrevStr,"Ln") == 0) moveAbbrev=LION;
	else cerr << "Unknown piece type in PieceRepresentation: " << abbrevStr << "###" << endl;

	env->ReleaseStringUTFChars(abbrev,abbrevStr);  
	if (env->ExceptionOccurred()) {
	  cerr << "Exception Occurred trying to ReleaseStringUTFChars for abbrevStr" << endl;
	  env->ExceptionDescribe();
	  env->ExceptionClear();
	  return NULL;
	}
	PieceCharacteristics *character=(homeBoard.getPieces())[moveAbbrev];

	Piece* newPiece=new Piece(character,colour,promoted,delayed,OK);
	if (newPiece == NULL) cerr << "newPiece allocation failed" << endl;
        //else {
	//  if (first) cout << "First allocation is at " << newPiece << endl;
	//}
        first=false;
	cell.set(newPiece);
      }
    }
  }

  // now to create a GenerateMove

  GeneratedMove firstMove(humanMove);
  // finally we can set up the search

  // now calculate the hashcode and checksum for thr initial position

  jlong hashcode=hash(homeBoard);
  jint checkSum=checksum(homeBoard);

  jfieldID            repID;

  repID=env->GetFieldID(playerClass,"repetitions","Z");
  if (repID == NULL) {
    cerr << "Unable to find field: 'repetitions' in Player" << endl;
    return NULL;
  }

  jboolean repetitions=env->GetBooleanField(obj,repID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get repetitions in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  jfieldID            matingModeID;

  matingModeID=env->GetFieldID(playerClass,"matingMode","Z");
  if (matingModeID == NULL) {
    cerr << "Unable to find field: 'matingMode' in Player" << endl;
    return NULL;
  }

  jboolean matingMode=env->GetBooleanField(obj,matingModeID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get matingMode in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  jfieldID            logID;

  logID=env->GetFieldID(playerClass,"log","Z");
  if (logID == NULL) {
    cerr << "Unable to find field: 'log' in Player" << endl;
    return NULL;
  }

  jboolean log=env->GetBooleanField(obj,logID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get log in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  jfieldID            detailedID;
  
  detailedID=env->GetFieldID(playerClass,"detailed","I");
  if (detailedID == NULL) {
    cerr << "Unable to find field: 'detailed' in Player" << endl;
    return NULL;
  }

  jint detailed=env->GetIntField(obj,detailedID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get detailed in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  jfieldID            logFileNameID;
    
  logFileNameID=env->GetFieldID(playerClass,"logFileName","Ljava/lang/String;");
  if (logFileNameID == NULL) {
    cerr << "Unable to find field: 'logFileName' in Player" << endl;
    return NULL;
  }

  jstring logFileName=static_cast<jstring>(env->GetObjectField(obj,logFileNameID));
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get logFileName in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  const char *logFileNameStr=env->GetStringUTFChars(logFileName, 0);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to GetStringUTFChars for logFileNameStr" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  jfieldID            plyID;
  
  plyID=env->GetFieldID(playerClass,"ply","I");
  if (plyID == NULL) {
    cerr << "Unable to find field: 'ply' in Player" << endl;
    return NULL;
  }

  jint ply=env->GetIntField(obj,plyID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get ply in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  jfieldID            t1ID;
  
  t1ID=env->GetFieldID(playerClass,"threshold1","I");
  if (plyID == NULL) {
    cerr << "Unable to find field: 'threshold1' in Player" << endl;
    return NULL;
  }

  jint threshold1=env->GetIntField(obj,t1ID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get threshold1 in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  if (matingMode) {
    alwaysPromote=false;
    repetitions=false; // but will be checked for Black's moves
  }
  Search theSearch(&firstMove, &homeBoard, colour, alwaysPromote, repetitions, 
		   log, detailed, matingMode);

  if (log){
    theSearch.openLogFile(logFileNameStr);
    theSearch.logFile << "<chuLog>" << endl;
  }
  env->ReleaseStringUTFChars(logFileName,logFileNameStr);  
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to ReleaseStringUTFChars for logFileNameStr" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  jint maximumDepth=1; // for iterative deepening
  jint value=0;
  
  double elapsed = 0;

  if (matingMode) {
    theSearch.setMaxDepth(ply); /* no iterative deepening */
    value=theSearch.search(0, ply, false, false, Search::MINUSINFINITY,
			   Search::INFINITY, firstMove, 0, hashcode, checkSum, false);
    maximumDepth=ply; // for statistics
    getStats(theSearch);
  }
  else {
    /*
     * now we need to repeatedly invoke search and increasing values of maximumDepth
     * until the time criteria are fulfilled
     */

    // first set up a SIGALRM handler

    signal(SIGALRM, catch_alarm);

    // next set up an alarm so we will not overrun the maximum time limit

    timeExhausted=false;
    alarm(maximumTime);

    bool timeUp=false;
    clock_t startTime = clock();
    jint moveNumber=humanMove.getNumber();
    bool earlyOpening=moveNumber < threshold1;
    jint previousValue=0;

    while (!timeUp) {
      previousValue=value;      
      theSearch.setMaxDepth(maximumDepth);
      if (log) theSearch.logFile << "<iteration num=\"" << maximumDepth << "\">" << endl;
      value=theSearch.search(0, maximumDepth, false, false, Search::MINUSINFINITY,
			     Search::INFINITY, firstMove, 0, hashcode, checkSum, true);
     
      clock_t endTime = clock();
      elapsed = double(endTime - startTime) / (CLOCKS_PER_SEC / 1000); // milliseconds 
      
      if (timeExhausted) {
	if (log) theSearch.logFile << "<interrupted/>" << endl;
	timeUp=true;
      }
      else {
	getStats(theSearch);

	if (earlyOpening && maximumDepth == ply) timeUp=true;
	else if (elapsed > basicTime) {
	  //if (value < (previousValue - 100)) maximumDepth++; // trouble looming
	  /*else*/ timeUp=true; // no trouble on the horizon
	}
	else maximumDepth++;
      }
      if (log) {
	if (!timeExhausted) {
	  char buffer[30];
	  theSearch.getPrincipalContinuation(0).notate(buffer);
	  theSearch.logFile << "<principalContinuation value=\"" << value <<  "\">" 
			    << buffer << "</principalContinuation>" << endl;
	}
	theSearch.logFile << "</iteration>" << endl;
      }
    }
  }

  alarm(0); // we don't want to be interrupted now

  if (log) {
    theSearch.logFile << "</chuLog>" << endl;
    theSearch.closeLogFile();
  }

  if (logStats) {
    statsFile << maximumDepth << " " << timeExhausted << " " 
	      << theSearch.getGreatestDepth() << " " 
	      << theSearch.getNodesSearched() << " " << elapsed << endl;
    statsFile.flush();
  }

  // now find out if the user wants to know the principalContinuation

  jfieldID  showpcID;

  showpcID=env->GetFieldID(playerClass,"showpc","Z");
  if (showpcID == NULL) {
    cerr << "Unable to find field: 'showpc' in Player" << endl;
    return NULL;
  }

  jboolean showpc=env->GetBooleanField(obj,showpcID);
  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred trying to get showpc in Player" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  if (showpc) {
    
    jfieldID  pcID;

    jsize arraySize=(matingMode ? (value < 0 ? 1 : 1+ply-value) : 1+maximumDepth);
    jclass stringClass=env->FindClass("java/lang/String");
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to find class String" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }
    jobjectArray array=env->NewObjectArray(arraySize,stringClass,NULL);
    if (array==NULL)
      cerr << "principalContinuation array cannot be constructed" << endl;
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to construct principalContinuation array" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }

    pcID=env->GetFieldID(playerClass,"principalContinuation","[Ljava/lang/String;");
    if (pcID == NULL) {
      cerr << "Unable to find field: 'principalContinuation' in Player" << endl;
      return NULL;
    }

    env->SetObjectField(obj,pcID,array);
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to set principalContinuation in Player" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }    

    char buffer[44];
    ostrstream streamBuffer(buffer,44);  

    if (matingMode) {  
      if (value == 0)
	streamBuffer << "Found no mate after " << ply << " moves\n" <<
	  "Best try is:" << ends;
      else if (value == -1) 
	streamBuffer << "This problem has no solution" << ends;
      else streamBuffer << "Found a mate in " << (ply - value) << ":" << ends;
    }

    else streamBuffer << "Found the following principal continuation:" << ends;

    env->SetObjectArrayElement(array,0,env->NewStringUTF(buffer));
    if (env->ExceptionOccurred()) {
      cerr << "Exception Occurred trying to set principalContinuation header line" << endl;
      env->ExceptionDescribe();
      env->ExceptionClear();
      return NULL;
    }  

    for (jint index=0; index<arraySize-1;index++) {
      char buffer[30];
      theSearch.getPrincipalContinuation(index).notate(buffer);
      env->SetObjectArrayElement(array,index+1,env->NewStringUTF(buffer));
      if (env->ExceptionOccurred()) {
	cerr << "Exception Occurred trying to set principalContinuation[" << index+1 <<"]"<< endl;
	env->ExceptionDescribe();
	env->ExceptionClear();
	return NULL;
      }
      if (theSearch.getPrincipalContinuation(index).wasInvalid()) break;
    }
  } // showpc

  // now we need to build a ComputerResult to return

  // first we need to build a Move jobject
  // we already have java objects set up, which we can reuse, as the previousMove
  // was replicated to ensure it is junk

  number=bestMove.getNumber();
  string abbrevString;

  switch (bestMove.getAbbrev()) {
  case NONE: cerr << "bestMove has abbrev of NONE" << endl; break; 
  case PAWN: abbrevString=string("P"); break;
  case TOKIN: abbrevString=string("+P"); break;
  case GOBETWEEN: abbrevString=string("GB"); break;
  case PrGOBETWEEN: abbrevString=string("+GB"); break;
  case COPPER: abbrevString=string("C"); break;
  case PrCOPPER: abbrevString=string("+C"); break;
  case SILVER: abbrevString=string("S"); break;
  case PrSILVER: abbrevString=string("+S"); break;
  case GOLD: abbrevString=string("G"); break;
  case PrGOLD: abbrevString=string("+G"); break;
  case LEOPARD: abbrevString=string("FL"); break;
  case PrLEOPARD: abbrevString=string("+FL"); break;
  case DRUNKELEPHANT: abbrevString=string("DE"); break;
  case PrDRUNKELEPHANT: abbrevString=string("+DE"); break;
  case BLINDTIGER: abbrevString=string("BT"); break;
  case FLYINGSTAG: abbrevString=string("+BT"); break;
  case KING: abbrevString=string("K"); break;
  case LANCE: abbrevString=string("L"); break;
  case WHITEHORSE: abbrevString=string("+L"); break;
  case REVERSECHARIOT: abbrevString=string("RC"); break;
  case WHALE: abbrevString=string("+RC"); break;
  case SIDEMOVER: abbrevString=string("SM"); break;
  case FREEBOAR: abbrevString=string("+SM"); break;
  case VERTICALMOVER: abbrevString=string("VM"); break;
  case FLYINGOX: abbrevString=string("+VM"); break;
  case KYLIN: abbrevString=string("Ky"); break;
  case PrKYLIN: abbrevString=string("+Ky"); break;
  case PHOENIX: abbrevString=string("Ph"); break;
  case PrPHOENIX: abbrevString=string("+Ph"); break;
  case BISHOP: abbrevString=string("B"); break;
  case PrBISHOP: abbrevString=string("+B"); break;
  case ROOK: abbrevString=string("R"); break;
  case PrROOK: abbrevString=string("+R"); break;
  case HORSE: abbrevString=string("DH"); break;
  case FALCON: abbrevString=string("+DH"); break;
  case DRAGON: abbrevString=string("DK"); break;
  case EAGLE: abbrevString=string("+DK"); break;
  case FREEKING: abbrevString=string("FK"); break;
  case LION: abbrevString=string("Ln"); break;
  }

  jstring abbrevJstring=env->NewStringUTF(abbrevString.c_str());

  const Coordinates &sourceCoords=bestMove.getSource();
  const Coordinates &destCoords=bestMove.getStage1();
  const Coordinates &dest2Coords=bestMove.getStage2();
  
  env->SetIntField(source,XID,sourceCoords.x());
  env->SetIntField(source,YID,sourceCoords.y());
  env->SetIntField(destination,XID,destCoords.x());
  env->SetIntField(destination,YID,destCoords.y());
  env->SetIntField(destination2,XID,dest2Coords.x());
  env->SetIntField(destination2,YID,dest2Coords.y());

  destinationSquares=jbyte(bestMove.squares());
  firstCapture=bestMove.wasFirstCaptured();
  secondCapture=bestMove.wasSecondCaptured();
  promotion=bestMove.wasPromoted();
  mcolour=bestMove.isBlack();
  lionSource=bestMove.isLionSource();
  lionTarget=bestMove.isLionTarget();
  doubleCapture=bestMove.isDoubleCapture();
  lionCaptured=bestMove.wasLionCaptured();
  didNotPromote=bestMove.didntPromote();
  promotionOK=bestMove.isOK();
  delayed=bestMove.wasDelayed();

  // now to actually build a Move jobject

  jmethodID createMoveID=env->GetMethodID(moveClass,"<init>",
	    "(ILjava/lang/String;LCoordinates;LCoordinates;LCoordinates;BZZZZZZZZZZZZZZZ)V");
  jobject newMove=env->NewObject(moveClass,createMoveID,number,abbrevJstring,source,
				 destination,destination2,destinationSquares,jboolean(false),
				 firstCapture,secondCapture,promotion,mcolour,lionSource,
				 jboolean(false),jboolean(false),doubleCapture,lionCaptured,
				 lionTarget,didNotPromote,jboolean(false),promotionOK,delayed);
  if (newMove == NULL)
    cerr << "Move constructor failed" << endl;

  // now build the result object and return

  jclass resultClass=env->FindClass("ComputerResult");

  jmethodID createResultID=env->GetMethodID(resultClass,"<init>",
			    "(IIIIIIIIIIIIILMove;)V");
  jobject newResult=env->NewObject(resultClass,createResultID,value,count,exactHits,
				   lowerCutoffs,upperCutoffs,nearMisses,hashingCollisions,
				   hashingErrors,replacements,repetitionCount,Search::cached,
				   hashEntries,maximumDepth,newMove);
  if (newResult == NULL)
    cerr << "ComputerResult constructor failed" << endl;

  if (env->ExceptionOccurred()) {
    cerr << "Exception Occurred in return processing from player.cpp" << endl;
    env->ExceptionDescribe();
    env->ExceptionClear();
    return NULL;
  }

  return newResult;
}


void catch_alarm (int sig) {
  timeExhausted=true;
}


void getStats(Search& theSearch) {
  count=theSearch.getNodesSearched();
  exactHits=theSearch.getExactHits();
  upperCutoffs=theSearch.getUpperCutoffs();
  lowerCutoffs=theSearch.getLowerCutoffs();
  nearMisses=theSearch.getNearMisses();
  hashingCollisions=theSearch.getHashingCollisions();
  hashingErrors=theSearch.getHashingErrors();
  replacements=theSearch.getReplacements();
  repetitionCount=theSearch.getRepetitions();

  bestMove=theSearch.getBestMove();
}











