/** -*- mode: C++ -*-
 * piece.h defines a lightweight piece (as used by the computer player)
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef PIECE_H 
#define PIECE_H
#include <jni.h>
#include "coordinates.h"
#include "directions.h"

enum PieceType {NONE, PAWN, TOKIN, GOBETWEEN, PrGOBETWEEN, COPPER, PrCOPPER, SILVER, PrSILVER,
		GOLD, PrGOLD, LEOPARD, PrLEOPARD, DRUNKELEPHANT, PrDRUNKELEPHANT, KING, 
		LANCE, WHITEHORSE, REVERSECHARIOT, WHALE, BLINDTIGER,
		FLYINGSTAG, KYLIN, PrKYLIN, PHOENIX, PrPHOENIX, BISHOP, PrBISHOP,
		SIDEMOVER, FREEBOAR, VERTICALMOVER, FLYINGOX, ROOK, PrROOK, HORSE, FALCON,
		DRAGON, EAGLE, FREEKING, LION};

class Piece; // forward declaration for friend statement
class Promotion;

class PieceCharacteristics {
  PieceType           type;       
  PieceType      promotion;
  jint            range[8];       // one for each direction
  bool             lion[8];       // one for each direction
  bool             jump[8];       // one for each direction
  bool           lionPower;
  bool           finalRank;
  bool         crownPrince;
  jint               value;

  friend class Piece;
 public:
  PieceCharacteristics(PieceType t, PieceType p, 
		      jint r0, jint r1, jint r2, jint r3, jint r4, jint r5, jint r6, jint r7,
		      bool j0, jint j1, jint j2, jint j3, jint j4, jint j5, jint j6, jint j7,
		      bool l0, jint l1, jint l2, jint l3, jint l4, jint l5, jint l6, jint l7,
		      bool lp, bool fr, bool cp, jint v
		      ) {
    type = t; promotion = p;
    range[0] = r0; range[1] = r1; range[2] = r2; range[3] = r3; range[4] = r4;
    range[5] = r5; range[6] = r6; range[7] = r7;
    jump[0] = j0; jump[1] = j1; jump[2] = j2; jump[3] = j3; jump[4] = j4;
    jump[5] = j5; jump[6] = j6; jump[7] = j7;
    lion[0] = l0; lion[1] = l1; lion[2] = l2; lion[3] = l3; lion[4] = l4;
    lion[5] = l5; lion[6] = l6; lion[7] = l7;
    lionPower = lp; crownPrince = cp; finalRank = fr; value = v;
  }
  PieceType getAbbrev() const { return type; } 
};

extern PieceCharacteristics *thePieces[];


class Piece {
  PieceCharacteristics *thePiece;
  bool                    colour;      // true == Black
  bool                  promoted;
  bool          delayedPromotion;
  bool               promotionOK;
  PieceType             demotion;
 public:
  Piece() { thePiece=0; }
  Piece(PieceCharacteristics *c, bool col, bool p, bool d, bool o)
    {thePiece=c; colour=col; promoted=p; delayedPromotion=d, promotionOK=o; demotion=NONE; }
  bool isBlack() const { return colour; }
  bool isPromoted() const { return promoted; }
  bool isDelayed() const { return delayedPromotion; }
  bool isOK() const { return promotionOK; }
  PieceType getAbbrev() const { return thePiece->getAbbrev(); } 
  PieceType getPromotion() const { return thePiece->promotion; } 
  PieceType getType() const { return thePiece->type; } 
  PieceType getDemotion() const { return demotion; } 
  PieceCharacteristics *getCharacteristics() const { return thePiece; } 
  jint getValue() const { return thePiece->value; }
  bool getColour() const { return colour; }
  jint getRange(Directions d) const { return thePiece->range[d]; }
  bool getJump(Directions d) const { return thePiece->jump[d]; }
  bool getLion(Directions d) const { return thePiece->lion[d]; }
  bool getLionPower() const { return thePiece->lionPower; }
  bool isCrownPrince() const { return thePiece->crownPrince; }
  void setOK(bool b) { promotionOK=b; }
  void setDelayed(bool b) { delayedPromotion=b; }
  void setDemotion(PieceType b) { demotion=b; }
  void setPromoted(bool b) { promoted=b; }
  void setColour(bool b) { colour=b; }
  void setCharacteristics(PieceCharacteristics *c) { thePiece=c; }
  void checkForPromotion(Promotion&,const Coordinates&, const Coordinates&,bool,
			 PieceCharacteristics*const*);
  void getAbbrevString(char *s) const { 
    switch (thePiece->getAbbrev()) {
    case NONE: strcpy(s,""); break;
    case PAWN: strcpy(s,"P"); break;
    case TOKIN: strcpy(s,"+P"); break;
    case GOBETWEEN: strcpy(s,"GB"); break;
    case PrGOBETWEEN: strcpy(s,"+GB"); break;
    case COPPER: strcpy(s,"C"); break;
    case PrCOPPER: strcpy(s,"+C"); break;
    case SILVER: strcpy(s,"S"); break;
    case PrSILVER: strcpy(s,"+S"); break;
    case GOLD: strcpy(s,"G"); break;
    case PrGOLD: strcpy(s,"+G"); break;
    case LEOPARD: strcpy(s,"FL"); break;
    case PrLEOPARD: strcpy(s,"+FL"); break;
    case DRUNKELEPHANT: strcpy(s,"DE"); break;
    case PrDRUNKELEPHANT: strcpy(s,"+DE"); break;
    case KING: strcpy(s,"K"); break;
    case LANCE: strcpy(s,"L"); break;
    case WHITEHORSE: strcpy(s,"+L"); break;
    case REVERSECHARIOT: strcpy(s,"RC"); break;
    case WHALE: strcpy(s,"+RC"); break;
    case BLINDTIGER: strcpy(s,"BT"); break;
    case FLYINGSTAG: strcpy(s,"+BT"); break;
    case KYLIN: strcpy(s,"Ky"); break;
    case PrKYLIN: strcpy(s,"+Ky"); break;
    case PHOENIX: strcpy(s,"Ph"); break;
    case PrPHOENIX: strcpy(s,"+Ph"); break;
    case BISHOP: strcpy(s,"B"); break;
    case PrBISHOP: strcpy(s,"+B"); break;
    case ROOK: strcpy(s,"R"); break;
    case PrROOK: strcpy(s,"+R"); break;
    case SIDEMOVER: strcpy(s,"SM"); break;
    case FREEBOAR: strcpy(s,"+SM"); break;
    case VERTICALMOVER: strcpy(s,"VM"); break;
    case FLYINGOX: strcpy(s,"+VM"); break;
    case HORSE: strcpy(s,"DH"); break;
    case FALCON: strcpy(s,"+DH"); break;
    case DRAGON: strcpy(s,"DK"); break;
    case EAGLE: strcpy(s,"+DK"); break;
    case FREEKING: strcpy(s,"FK"); break;
    case LION: strcpy(s,"Ln"); break;
    }
    /*return string(s);*/ } 

  void promote(PieceCharacteristics *p[]) {
    PieceType pr=thePiece->promotion;
    thePiece=p[pr];
    promoted=true;
    demotion=pr;
    delayedPromotion=false;
    promotionOK=false;
  }

  void demote(PieceCharacteristics *p[]) {
    promoted=false;
    thePiece=p[demotion]; // other fields must be obtained from the PieceSaving
    demotion=NONE;
  }
};


struct PieceSaving {
  bool                  promoted;
  bool                    colour;
  bool          delayedPromotion;
  bool               promotionOK;
  PieceType             demotion;
  PieceCharacteristics *chars;
  void set(Piece *p) {
    promoted=p->isPromoted();
    colour=p->isBlack();
    chars=p->getCharacteristics();
    if (!chars) cerr << "get chars failed" << endl;
    delayedPromotion=p->isDelayed();
    promotionOK=p->isOK();
    demotion=p->getType();
  }
  void restore(Piece *p) {
    p->setPromoted(promoted);
    p->setColour(colour);
    if (!chars) cerr << "chars is null" << endl;
    p->setCharacteristics(chars);
    p->setDelayed(delayedPromotion);
    p->setOK(promotionOK);
    p->setDemotion(demotion);
  }
};
struct Targets {
  PieceSaving source;
  PieceSaving target;
  PieceSaving target2;
};
#endif

























