/** -*- mode: C++ -*-
 * move.h defines a move (as used by the computer player)
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef MOVE_H 
#define MOVE_H
#include <jni.h>
#include <string>
#include "piece.h"
#include "coordinates.h"

class Move {
  jint                  number; 
  PieceType             abbrev;  
  Coordinates           source;
  Coordinates      destination;
  Coordinates     destination2;
  jint      destinationSquares;
  bool            firstCapture;
  bool           secondCapture;
  bool               promotion;  // will piece promote at end of move?
  bool                   setOK;  // tell updatePosition to set promotionOK 
  bool              setDelayed;  // tell updatePosition to set promotionDelayed 
  bool                  colour;  // true is Black
  bool              lionSource;
  bool     anotherMovePossible;
  bool           doubleCapture;  // set at move 1 if a Lion can be captured on a B square on move 2
  bool            lionCaptured;
  bool              lionTarget;
  bool           didNotPromote;
  bool                invalid;   // if true, cannot use this principal continuation move
  void                 reset();
public:  
  Move() { number=0; colour=true; reset(); }
  Move(jint,PieceType,const Coordinates&,const Coordinates&,const Coordinates&,jint,
       bool,bool,bool,bool,bool,bool,bool,bool,bool,bool,bool,bool);
  Move(Move const&);
  Move& replicate(const Move&);
  void notate(char*) const;
  void validate() { invalid = false; }
  bool wasFirstCaptured() const { return firstCapture; }
  bool wasSecondCaptured() const { return secondCapture; }
  bool wasLionCaptured() const { return lionCaptured; }
  Coordinates const& getSource() const { return source; } 
  Coordinates const& getStage1() const { return destination; } 
  Coordinates const& getStage2() const { return destination2; } 
  bool isBlack() const { return colour; }
  bool isLionTarget() const { return lionTarget; }
  bool isLionSource() const { return lionSource; }
  bool isDoubleCapture() const { return doubleCapture; }
  bool isOK() const { return setOK; }
  bool wasDelayed() const { return setDelayed; }
  bool wasInvalid() const { return invalid; }
  bool isAnotherMovePossible() const { return anotherMovePossible; }
  bool wasPromoted() const { return promotion; } 
  bool didntPromote() const { return didNotPromote; } 
  bool wasPassOrIgui() const { 
    return (destinationSquares == 2 && destination2 == source); }
  PieceType const& getAbbrev() const { return abbrev; }
  jint squares() const { return destinationSquares; }
  jint getNumber() const { return number; }
  void setSource(Coordinates const &c) { source.set(c); }
  void setSquares(jint n) { destinationSquares=n; }
  void setPromoted() { promotion = true; }
  void setColour(bool b) { colour=b; }
  void setNumber(int n) { number=n; }
  void setDidNotPromote() { promotion = false; didNotPromote=true; }
  void setLionSource() { lionSource = true; }
  void setLionTarget() { lionTarget = true; }
  void setLionCaptured() { lionCaptured = true; }
  void setDoubleCapture() { doubleCapture = true; }
  void setFirstCapture() { firstCapture = true; }
  void setSecondCapture() { secondCapture = true; }
  void setDestination1(const Coordinates &c) { destination = c; }
  void setDestination2(const Coordinates &c) { destination2 = c; }
  void setAnotherMovePossible() { anotherMovePossible = true; }
  void forceDoubleMove() { destinationSquares=2; }
  void setAbbrev(const PieceType &t) { abbrev=t; }
  void setPromotionOK() { setOK=true; }
  void setPromotionDelayed() { setDelayed=true; }
  void invalidate() { invalid=true; }
  bool operator==(const Move &m) const {
    if ( invalid || m.invalid) return false; // invalid moves are not equal to anything
    else if (colour != m.colour) return false;
    else if (abbrev != m.abbrev) return false;
    else if (promotion != m.promotion) return false;
    else if (firstCapture != m.firstCapture) return false;
    else if (destinationSquares != m.destinationSquares) return false;
    else if (source != m.source) return false;
    else if (destination != m.destination) return false;
    else if (destinationSquares == 2 && ((destination2 != m.destination2) ||
					 secondCapture != m.secondCapture)) return false;
    else return true; // everything vital matches (we can ignore number, which must be ok)
    }
};

#endif

