/**
 * move.cpp implements the substantive parts of a Move (as used by the computer player)
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

using namespace std;

#ifdef INLINE_OK 
#define INLINE inline
#else
#define INLINE  
#endif
#include "move.h"
#include <strstream.h>
#include <cstring>

/**
 * Notate builds a string that fully represents the move in the client context
 *
 */

INLINE void Move::notate(char *c) const
{
  bool debug = false;
  string description;

  // first the move number, followed by a blank

  char numberBuffer[6];
  ostrstream numberStreamBuffer(numberBuffer,6);  
  numberStreamBuffer << number << ' ' << ends ;

  description = numberBuffer; 

  if (invalid) {
    description+="invalid continuation"; 
    strcpy(c,description.c_str());
    return;
  }
  // now look up the abbreviation

  string abbrevString;

  switch (abbrev) {
  case NONE: 
    cerr << "Move has no abbreviation" << endl;
    abbrevString=string("NN");
    debug=true;
    break;
  case PAWN: abbrevString=string("P "); break;
  case TOKIN: abbrevString=string("+P "); break;
  case GOBETWEEN: abbrevString=string("GB "); break;
  case PrGOBETWEEN: abbrevString=string("+GB "); break;
  case COPPER: abbrevString=string("C "); break;
  case PrCOPPER: abbrevString=string("+C "); break;
  case SILVER: abbrevString=string("S "); break;
  case PrSILVER: abbrevString=string("+S "); break;
  case GOLD: abbrevString=string("G "); break;
  case PrGOLD: abbrevString=string("+G "); break;
  case LEOPARD: abbrevString=string("FL "); break;
  case PrLEOPARD: abbrevString=string("+FL "); break;
  case DRUNKELEPHANT: abbrevString=string("DE "); break;
  case PrDRUNKELEPHANT: abbrevString=string("+DE "); break;
  case KING: abbrevString=string("K "); break;
  case LANCE: abbrevString=string("L "); break;
  case WHITEHORSE: abbrevString=string("+L "); break;
  case REVERSECHARIOT: abbrevString=string("RC "); break;
  case WHALE: abbrevString=string("+RC "); break;
  case BLINDTIGER: abbrevString=string("BT "); break;
  case FLYINGSTAG: abbrevString=string("+BT "); break;
  case KYLIN: abbrevString=string("Ky "); break;
  case PrKYLIN: abbrevString=string("+Ky "); break;
  case PHOENIX: abbrevString=string("Ph "); break;
  case PrPHOENIX: abbrevString=string("+Ph "); break;
  case BISHOP: abbrevString=string("B "); break;
  case PrBISHOP: abbrevString=string("+B "); break;
  case SIDEMOVER: abbrevString=string("SM "); break;
  case FREEBOAR: abbrevString=string("+SM "); break;
  case VERTICALMOVER: abbrevString=string("VM "); break;
  case FLYINGOX: abbrevString=string("+VM "); break;
  case ROOK: abbrevString=string("R "); break;
  case PrROOK: abbrevString=string("+R "); break;
  case HORSE: abbrevString=string("DH "); break;
  case FALCON: abbrevString=string("+DH "); break;
  case DRAGON: abbrevString=string("DK "); break;
  case EAGLE: abbrevString=string("+DK "); break;
  case FREEKING: abbrevString=string("FK "); break;
  case LION: abbrevString=string("Ln "); break;
  }
  description += abbrevString; 
  char sourceBuffer[5];
  char destBuffer[5];
  char dest2Buffer[5];
  source.describe(sourceBuffer);
  description += string(sourceBuffer);

  // pass

  if (destinationSquares == 2 && !firstCapture) description += "! ";

  else { // not a pass
    if (destinationSquares == 2 && destination2 == source) { //igui 
      destination.describe(destBuffer);
      description += "x! " + string(destBuffer);
      strcpy(c,description.c_str());
      return;
    } //igui 
    if (firstCapture) description += "x ";
    else description += "- ";
    destination.describe(destBuffer);
    description += string(destBuffer);
    if (destinationSquares == 2) { // double move
      if (secondCapture) description += "x ";
      else description += "- ";
      destination2.describe(dest2Buffer);
      description += string(dest2Buffer);
    } // double move
    if (promotion) description += "+";
    else if (didNotPromote) description += "=";
  } // not a pass

  strcpy(c,description.c_str());
  if (debug) cerr << description << endl;
}


INLINE Move& Move::replicate(const Move &m)
{
  reset();
  if (m.colour) number=m.number;
  else number=m.number+1;
  colour=!m.colour;

  return *this;
}


INLINE void Move::reset()
{
  destinationSquares=1;
  promotion=false;
  firstCapture=false;
  secondCapture=false;
  anotherMovePossible=false;
  lionSource=false;
  lionTarget=false;
  doubleCapture=false;
  lionCaptured=false;
  didNotPromote=false;
  abbrev=NONE;
  source.reset();
  destination.reset();
  destination2.reset();
  setOK=false;
  setDelayed=false;
  invalid=false;
}



INLINE Move::Move(const Move &m)
{
  number=m.number; 
  abbrev=m.abbrev;  
  source=m.source;
  destination=m.destination;
  destination2=m.destination2;
  destinationSquares=m.destinationSquares;
  firstCapture=m.firstCapture;
  secondCapture=m.secondCapture;
  promotion=m.promotion;
  colour=m.colour;
  lionSource=m.lionSource;
  anotherMovePossible=m.anotherMovePossible;
  doubleCapture=m.doubleCapture;
  lionCaptured=m.lionCaptured;
  lionTarget=m.lionTarget;
  didNotPromote=m.didNotPromote;
  setOK=m.setOK;
  setDelayed=m.setDelayed;
  invalid=m.invalid;
}

INLINE Move::Move(jint n,PieceType t,const Coordinates&so, const Coordinates&de,
		  const Coordinates&de2,
		  jint sq,bool fc,bool sc,bool pr,bool ok,bool del,bool col,bool ls ,
		  bool amp,bool dc,
		  bool lc,bool lt,bool dnp)
{
  number=n; abbrev=t; source=so; destination=de; destination2=de2; destinationSquares=sq;
  firstCapture=fc; secondCapture=sc; promotion=pr; setOK=ok; setDelayed=del; colour=col;
  lionSource=ls; anotherMovePossible=amp; doubleCapture=dc; lionCaptured=lc; lionTarget=lt;
  didNotPromote=dnp; invalid=false;
}





