/**
 * generate.cpp implements the substantive parts of a Generator
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

using namespace std;

#ifdef INLINE_OK 
#define INLINE inline
#else
#define INLINE  
#endif
#include "generate.h"
#include <iostream>
#include <algorithm>

static int INLINE  lionDistance(Coordinates &a,Coordinates &b)
{
  if (abs(a.x()-b.x()) == 2) return 2;
  else if (abs(a.y()-b.y()) == 2) return 2;
  else return 1; // okay, okay, could be 0
}


INLINE Generator::Generator(Board *const h,const Move *m, bool e, bool a, killerListType *k)
{
  listOfMoves=listType();
  homeBoard=h; 
  previousMove=m;
  extraDeep=e;
  alwaysPromote=a;
  extraDeepOK=true;
  killers=k;
  
  if (extraDeep) {
    if (previousMove->wasSecondCaptured()) {
      previousFile=previousMove->getStage2().x();
      previousRank=previousMove->getStage2().y();
    }
    else if (previousMove->wasFirstCaptured()) {
      previousFile=previousMove->getStage1().x();
      previousRank=previousMove->getStage1().y();
    }
    else extraDeepOK = false;
  }
}


INLINE void Generator::generate()
{
  bool colour=!previousMove->isBlack();

  if (extraDeep && !extraDeepOK) return;

  for (jint rank=0; rank < 12; rank++)  // good candidate for register?
    for (jint file=11; file >= 0; file--) { // good candidate for register?
      const Cell &cell=homeBoard->getCell(file,rank);
      const Piece *sourcePiece=cell.getPiece();
      if (sourcePiece && sourcePiece->isBlack() == colour) { // colour OK
	Coordinates sourceCoordinates=Coordinates(file,rank);
	Move newMove; // pattern for all the moves to be generated
	newMove.replicate(*previousMove); // this changes the colour, etc.
	newMove.setSource(sourceCoordinates);
	PieceType abbrev=sourcePiece->getAbbrev();
	newMove.setAbbrev(abbrev);
	if (abbrev == LION || abbrev == PrKYLIN) newMove.setLionSource();
	// now set the lighting to determine a super-set of the legal moves

	homeBoard->setLighting(sourceCoordinates);

	// and try every square on the board
	for (jint destRank=0; destRank < 12; destRank++) // good candidate for register?
	  for (jint destFile=11; destFile >= 0; destFile--) { // good candidate for register?
	    if (file == destFile && rank == destRank) ; // no pass moves are generated
	    else {
	      Cell &destCell=homeBoard->getCell(destFile,destRank);
              
	      const Piece *targetPiece=destCell.getPiece();
	      if (extraDeep) {
		// we only look at final captures on the previous square
		if (targetPiece && (previousFile==destFile) && (previousRank==destRank)); // OK
		else continue; // wasn't the right square
	      }
	      
	      Lighting light=destCell.getLighting();
	      if (light != CLEAR) { // a move!
		jint gain=0;
		Move thisMove(newMove); // clone the move skeleton
		Promotion promotion; // to get the results
		promotion.reset(); // I'm not sure if this is needed
		Coordinates destinationCoordinates(destFile, destRank);
		bool legal=generateNextMove(promotion, thisMove, sourceCoordinates, 
					    destinationCoordinates);
		bool moreMoves=false;
		if (legal) {
		  if (thisMove.wasFirstCaptured()) gain=targetPiece->getValue();
		  if (promotion.wasCompulsory()) { // only 1 possible promotion, and no repetition
		    thisMove.setPromoted();
		    // check the killer list
		    if (killers && 
			find(killers->begin(),killers->end(),thisMove) != killers->end())
		      gain=KILLER_SCORE;
		    else gain+=promotion.gain();
		    // promotions go to the front of the list to aid the alpha-beta algorithm
		    listOfMoves.push_front(pair<GeneratedMove,jint>(
					    GeneratedMove(thisMove),gain));
		  }
		  else if (!promotion.wasPromoted()) { // not promoted
		    if (promotion.isOK()) thisMove.setPromotionOK(); // can promote next time
		    if (thisMove.wasFirstCaptured()) {
		      if (thisMove.isAnotherMovePossible()) moreMoves=true; // we suppress passes
		      // check the killer list
		      if (killers && 
			  find(killers->begin(),killers->end(),thisMove) != killers->end())
			gain=KILLER_SCORE;
		      // captures go to the front also
		      listOfMoves.push_front(pair<GeneratedMove,jint>(
					      GeneratedMove(thisMove),gain));
		    }
		    else { // not first capture
		      // check the killer list
		      if (killers &&
			  find(killers->begin(),killers->end(),thisMove) != killers->end()) {
			gain=KILLER_SCORE;
			listOfMoves.push_front(pair<GeneratedMove,jint>(
						GeneratedMove(thisMove),gain));
		      }
		      else listOfMoves.push_back(pair<GeneratedMove,jint>(
						GeneratedMove(thisMove),gain));
		    }
		  }
		  else { // two moves
		    Move promotedMove(thisMove); // clone the move
		    promotedMove.setPromoted();
		    jint promotedGain;
		    promotedGain=gain+promotion.gain();
		    if (killers &&
			find(killers->begin(),killers->end(),promotedMove) != killers->end())
		      promotedGain=KILLER_SCORE;
		    listOfMoves.push_front(pair<GeneratedMove,jint>(
					    GeneratedMove(promotedMove),promotedGain));
                    if (!alwaysPromote) {
		      thisMove.setDidNotPromote();
		      if (promotion.isOK()) thisMove.setPromotionOK(); // can promote next time
		      thisMove.setPromotionDelayed(); 
		      if (killers &&
			  find(killers->begin(),killers->end(),thisMove) != killers->end())
			gain=KILLER_SCORE;
		      if (thisMove.wasFirstCaptured()) // captures go to the front also
			listOfMoves.push_front(pair<GeneratedMove,jint>(
						GeneratedMove(thisMove),gain));
		      else {
			if (killers && 
			    find(killers->begin(),killers->end(),thisMove) != killers->end()) {
			  gain=KILLER_SCORE;
			  listOfMoves.push_front(pair<GeneratedMove,jint>(
						  GeneratedMove(thisMove),gain));
			}
			else listOfMoves.push_back(pair<GeneratedMove,jint>(
						    GeneratedMove(thisMove),gain));
		      }
		    }
		  }
		  if (moreMoves) { 
		    for (Directions dir=NORTH; dir <= SOUTHEAST; dir=Directions(dir + 1)) {
		      Move secondMove(thisMove); // clone the move
		      secondMove.forceDoubleMove();
		      abbrev=secondMove.getAbbrev();
		      secondGain=0; // this will be overridden by the generating function
		      Coordinates dest2Coords=secondMove.getStage1();
		      dest2Coords.inc(dir,1,secondMove.isBlack());
		      // check second move is on the board
		      if (dest2Coords.x() > 11 || dest2Coords.x() < 0 || 
			  dest2Coords.y() > 11 || dest2Coords.y() < 0)
			continue;
		      secondMove.setDestination2(dest2Coords);
		      legal=false;
		      switch (abbrev) {
		      case LION:
		      case PrKYLIN: legal=generateLion(secondMove);
			break;
		      case FALCON: legal=hornedFalcon(secondMove);
			break;
		      case EAGLE: legal=soaringEagle(secondMove);
			break;
		      default: 
			cerr << "Double move attempted by piece type " << int(abbrev) << endl;
		      }
		      // double moves  go to the front of the list to aid the alpha-beta algorithm
		      if (legal) listOfMoves.push_front(pair<GeneratedMove,jint>(
						       GeneratedMove(secondMove),gain+secondGain));
		    }
		  }
		  }
		}
	    }
	  }
      }	
    }
}

INLINE bool Generator::generateNextMove(Promotion &promotion, Move &thisMove, 
				 const Coordinates &sourceCoordinates, 
				 const Coordinates &targetCoordinates)
{
  thisMove.setDestination1(targetCoordinates);

  Piece *sourcePiece=homeBoard->getCell(sourceCoordinates.x(),sourceCoordinates.y()).getPiece();
  Piece *targetPiece=homeBoard->getCell(targetCoordinates.x(),targetCoordinates.y()).getPiece();

  if (targetPiece) {
    thisMove.setFirstCapture();
    PieceType targetAbbrev=targetPiece->getAbbrev();
    if (targetAbbrev == LION || targetAbbrev == PrKYLIN) thisMove.setLionTarget();
    if (!(targetAbbrev == PAWN || targetAbbrev == GOBETWEEN)) thisMove.setDoubleCapture();
  }
  bool legal=homeBoard->isLegal(&thisMove,*previousMove);
  if (legal) {
    if (thisMove.isLionTarget() && !thisMove.isLionSource())
      thisMove.setLionCaptured();
    // check for promotion
    if (sourcePiece->getPromotion() != NONE)  
      sourcePiece->checkForPromotion(promotion,sourceCoordinates,targetCoordinates,
				     thisMove.wasFirstCaptured(),homeBoard->getPieces());
  }

  return legal;
}

INLINE bool Generator::soaringEagle(Move &move)
{
  Coordinates origin=move.getSource();
  Coordinates here=move.getStage1();
  Coordinates there=move.getStage2();
  Piece *targetPiece=homeBoard->getCell(there.x(),there.y()).getPiece();
  bool legal=false;

  if (origin.x() == there.x() && origin.y() == there.y()) return true; // igui

  if ((origin.x() == here.x()+1) && (origin.x() == there.x()+2)
      && (origin.y() == here.y()+1) && (origin.y() == there.y()+2)) legal=true;
  else if ((origin.x() == here.x()-1) && (origin.x() == there.x()-2)
	   && (origin.y() == here.y()+1) && (origin.y() == there.y()+2)) legal=true;
  else if ((origin.x() == here.x()+1) && (origin.x() == there.x()+2)
	   && (origin.y() == here.y()-1) && (origin.y() == there.y()-2)) legal=true;
  else if ((origin.x() == here.x()-1) && (origin.x() == there.x()-2)
	   && (origin.y() == here.y()-1) && (origin.y() == there.y()-2)) legal=true;

  if (legal && targetPiece) {
    if (targetPiece->isBlack() == move.isBlack()) return false;
    else {
      move.setSecondCapture();
      secondGain=targetPiece->getValue();
      PieceType targetAbbrev=targetPiece->getAbbrev();
      if (targetAbbrev == LION || targetAbbrev == PrKYLIN) move.setLionCaptured();
    }
  }

  return legal;
}

INLINE bool Generator::hornedFalcon(Move &move)
{
  Coordinates origin=move.getSource();
  Coordinates here=move.getStage1();
  Coordinates there=move.getStage2();
  Piece *targetPiece=homeBoard->getCell(there.x(),there.y()).getPiece();
  bool legal=false;

  if (origin.x() == there.x() && origin.y() == there.y()) return true; // igui

  if ((origin.y() == here.y()+1) && (origin.y() == there.y()+2)
      && (origin.x() == here.x()) && (origin.x() == there.x())) legal=true;
  else if ((origin.y() == here.y()-1) && (origin.y() == there.y()-2)
      && (origin.x() == here.x()) && (origin.x() == there.x())) legal=true;

  if (legal && targetPiece) {
    if (targetPiece->isBlack() == move.isBlack()) return false;
    else {
      move.setSecondCapture();
      secondGain=targetPiece->getValue();
      PieceType targetAbbrev=targetPiece->getAbbrev();
      if (targetAbbrev == LION || targetAbbrev == PrKYLIN) move.setLionCaptured();
    }
  }

  return legal;
}

INLINE bool Generator::generateLion(Move &move)
{
  Coordinates origin=move.getSource();
  Coordinates there=move.getStage2();
  Cell &target2Cell=homeBoard->getCell(there.x(),there.y());
  Piece *targetPiece=target2Cell.getPiece();
  PieceType targetAbbrev=NONE;
  bool legal=true;

  if (origin.x() == there.x() && origin.y() == there.y()) return true; // igui

  if (targetPiece) {
    if (targetPiece->isBlack() == move.isBlack()) return false;
    else {
      move.setSecondCapture();
      secondGain=targetPiece->getValue();
    }
    targetAbbrev=targetPiece->getAbbrev();
  }


  // double capture
  if (move.wasFirstCaptured() && move.wasSecondCaptured() 
      && (targetAbbrev == LION || targetAbbrev == PrKYLIN)) {
    legal=move.isDoubleCapture();
    if (!legal) {
      if (lionDistance(origin,there) == 1) legal=true;
      else {
        // temporarily remove the source Lion from the board, to allow for the hidden
        // protector rule.
        Cell &sourceCell=homeBoard->getCell(origin.x(),origin.y());
	Piece *sourceLion=sourceCell.getPiece();
	sourceCell.reset();
	legal=homeBoard->isSquareUnprotected(there,!move.isBlack());
        // put the Lion back 
        sourceCell.set(sourceLion); 
      }
    }
  } 
  return legal;
}
