/** -*- mode: C++ -*-
 * board.h defines a lightweight board (as used by the computer player)
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef BOARD_H 
#define BOARD_H
#include <jni.h>
#include <set>
#include <string>
#include "cell.h"
#include "coordinates.h"
#include "move.h"

class Board;

class Forsyth {
  // this representation can be used to write the forsyth string
  jbyte forsyth[12][12];
public:
  Forsyth() {};
  void set(int file, int rank, jbyte value) { forsyth[file][rank]=value; }
  bool operator<(const Forsyth &f) const {
    for (int rank=0; rank < 12; rank++)
      for (int file=11; file > -1; file--) {
	bool less=(forsyth[file][rank] < f.forsyth[file][rank]);
        if (less) return true;  
	bool more=(forsyth[file][rank] > f.forsyth[file][rank]);
        if (more) return false;  
      }
    return false;
  }
  void describe(Board*);
};

typedef set<Forsyth> positionType; 

class Board {
  Cell          *cells[12][12];
  jint        piecesOnBoard[2];
  jint         crownPrinces[2];
  jint                   lions;
  positionType   *positions[2];
  PieceCharacteristics *thePieces[LION+1];
 public:
  Board(jint l);
  ~Board();
  void describeBoard(); // for debugging
  jint evaluate(bool);
  jint getCrownPrinces(bool c) const { return crownPrinces[c]; }
  PieceType getAbbrev(jint f, jint r) const { return cells[f][r]->getAbbrev();  }
  jint getValue(jint f, jint r) const { return cells[f][r]->getValue();  }
  bool getColour(jint f, jint r) const { return cells[f][r]->getColour();  }
  Cell& getCell(jint f, jint r) const { return *cells[f][r]; }
  PieceCharacteristics *const*getPieces() const { return thePieces; }
  void setLighting(Coordinates const&);
  void setInfluence(Coordinates const&, bool);
  void clearLighting();
  void backOutMove(Move&, Targets&, bool);
  void adjustCPsAfterCapture(Piece*);
  void adjustCPsAfterPromotion(Piece*);
  void adjustCPsAfterRestoration(Piece*);
  void adjustCPsBeforeDemotion(Piece*);
  void setPiecesOnBoard(jint p, jint i) { piecesOnBoard[i]=p; }
  void setCrownPrinces(jint c, jint i) { crownPrinces[i]=c; }
  bool addPosition(bool col) {
    Forsyth *f=forsyth(); 
    //f->describe(this);
    bool ok=(positions[col]->insert(*f)).second; 
    delete f;
    return ok;
  }
  bool checkPosition(bool col) {
    Forsyth *f=forsyth(); 
    bool ok=(positions[col]->find(*f) != positions[col]->end()); 
    delete f;
    return ok;
  }
  void addPosition(string, bool);
  void restorePosition(bool col) {
    Forsyth *f=forsyth(); 
    positions[col]->erase(*f);
    delete f;
  }
  bool isLegal(Move*,const Move&);
  bool isSquareUnprotected(const Coordinates &,bool);
  bool isCheck(bool);
  bool performMove(Move&,Targets&, bool);
  Forsyth *forsyth();
};

#endif
