/**
 * board.cpp implements a lightweight board (as used by the computer player)
 *
 * Copyright 1999 Colin P. Adams
 *
 * Email: colin@colina.demon.co.uk
 *  
 *   This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

using namespace std;

#ifdef INLINE_OK 
#define INLINE inline
#else
#define INLINE  
#endif
#include "board.h"
#include <strstream.h>
#include <cctype>

Board::Board(jint l)
{
  lions=l;
  piecesOnBoard[0] = 0;
  piecesOnBoard[1] = 0;
  crownPrinces[0]  = 0;
  crownPrinces[1]  = 0;
  (positions)[0] = new positionType();
  if ((positions)[0] == NULL) cerr << "positions[0] is null" << endl;
  (positions)[1] = new positionType();
  if ((positions)[1] == NULL) cerr << "positions[1] is null" << endl;

  // now build the cells

  for (jint rank=0; rank < 12; rank++)
    for (jint file=11; file >= 0; file--) {
      cells[file][rank] = new Cell();
  if (cells[file][rank] == NULL) cerr << "cell is null" << endl;
    }

  // now build the pieces table

  thePieces[NONE] = new  PieceCharacteristics(NONE, NONE, 
			  0, 0, 0, 0, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 0);

  thePieces[PAWN] = new  PieceCharacteristics(PAWN, TOKIN, 
			  1, 0, 0, 0, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, true, false, 100);
  if (thePieces[PAWN] == NULL) cerr << "PAWN is null" << endl;
  thePieces[TOKIN] = new  PieceCharacteristics(TOKIN, NONE, 
			  1, 1, 1, 1, 1, 1, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 270);
  if (thePieces[TOKIN] == NULL) cerr << "TOKIN is null" << endl;
  thePieces[GOBETWEEN] = new  PieceCharacteristics(GOBETWEEN, PrGOBETWEEN, 
			  1, 1, 0, 0, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false,  101);
  if (thePieces[GOBETWEEN] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrGOBETWEEN] = new  PieceCharacteristics(PrGOBETWEEN, NONE,
			  1, 0, 1, 1, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 280);
  if (thePieces[PrGOBETWEEN] == NULL) cerr << "Piece is null" << endl;
  thePieces[COPPER] = new  PieceCharacteristics( COPPER, PrCOPPER, 
			  1, 1, 0, 0, 1, 1, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 260);
  if (thePieces[COPPER] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrCOPPER] = new  PieceCharacteristics(PrCOPPER, NONE,
			  1, 1,12,12, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 500);
  if (thePieces[PrCOPPER] == NULL) cerr << "Piece is null" << endl;
  thePieces[SILVER] = new  PieceCharacteristics(SILVER, PrSILVER,
			  1, 0, 0, 0, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 350);
  if (thePieces[SILVER] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrSILVER] = new  PieceCharacteristics(PrSILVER, NONE,
			  12, 12, 1, 1, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 540);
  if (thePieces[PrSILVER] == NULL) cerr << "Piece is null" << endl;
  thePieces[GOLD] = new  PieceCharacteristics(GOLD, PrGOLD,
			  1, 1, 1, 1, 1, 1, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 600);
  if (thePieces[GOLD] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrGOLD] = new  PieceCharacteristics(PrGOLD, NONE,
			  12, 12, 12, 12, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 950);
  if (thePieces[PrGOLD] == NULL) cerr << "Piece is null" << endl;
  thePieces[LEOPARD] = new  PieceCharacteristics(LEOPARD, PrLEOPARD,
			  1, 1, 0, 0, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 400);
  if (thePieces[LEOPARD] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrLEOPARD] = new  PieceCharacteristics(PrLEOPARD, NONE,
			   0,  0,  0,  0,12,12,12,12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 560);
  if (thePieces[PrLEOPARD] == NULL) cerr << "Piece is null" << endl;
  thePieces[DRUNKELEPHANT] = new  PieceCharacteristics(DRUNKELEPHANT, PrDRUNKELEPHANT,
			  1, 0, 1, 1, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 580);
  if (thePieces[DRUNKELEPHANT] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrDRUNKELEPHANT] = new  PieceCharacteristics(PrDRUNKELEPHANT, NONE,
			   1,  1,  1,  1, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false,  true, 650);
  if (thePieces[PrDRUNKELEPHANT] == NULL) cerr << "Piece is null" << endl;
  thePieces[KING] = new  PieceCharacteristics(KING, NONE,
			   1,  1,  1,  1, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false,  true, 650);
  if (thePieces[KING] == NULL) cerr << "Piece is null" << endl;
  thePieces[LANCE] = new  PieceCharacteristics(LANCE, WHITEHORSE,
			  12, 0, 0, 0, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false,  true, false, 720);
  if (thePieces[LANCE] == NULL) cerr << "Piece is null" << endl;
  thePieces[WHITEHORSE] = new  PieceCharacteristics(WHITEHORSE, NONE,
			   12,  12,  0,  0, 12, 12, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 820);
  if (thePieces[WHITEHORSE] == NULL) cerr << "Piece is null" << endl;
  thePieces[REVERSECHARIOT] = new  PieceCharacteristics(REVERSECHARIOT, WHALE,
			  12, 12, 0, 0, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false,  false, false, 750);
  if (thePieces[REVERSECHARIOT] == NULL) cerr << "Piece is null" << endl;
  thePieces[WHALE] = new  PieceCharacteristics(WHALE, NONE,
			   12,  12,  0,  0, 0, 0, 12, 12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 800);
  if (thePieces[WHALE] == NULL) cerr << "Piece is null" << endl;
  thePieces[BLINDTIGER] = new  PieceCharacteristics(BLINDTIGER, FLYINGSTAG,
			  0, 1, 1, 1, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false,  false, false, 380);
  if (thePieces[BLINDTIGER] == NULL) cerr << "Piece is null" << endl;
  thePieces[FLYINGSTAG] = new  PieceCharacteristics(FLYINGSTAG, NONE,
			   12,  12,  1,  1, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 850);
  if (thePieces[FLYINGSTAG] == NULL) cerr << "Piece is null" << endl;
  thePieces[KYLIN] = new  PieceCharacteristics(KYLIN, PrKYLIN,
			  0, 0, 0, 0, 1, 1, 1, 1,
			   true,  true,  true,  true, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false,  false, false, 860);
  if (thePieces[KYLIN] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrKYLIN] = new  PieceCharacteristics(PrKYLIN, NONE,
			    0,   0,  0,  0, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					      true, false, false,4000);
  if (thePieces[PrKYLIN] == NULL) cerr << "Piece is null" << endl;
  thePieces[PHOENIX] = new  PieceCharacteristics(PHOENIX, PrPHOENIX,
			  1, 1, 1, 1, 0, 0, 0, 0,
			  false, false, false, false,  true,  true,  true,  true,
			  false, false, false, false, false, false, false, false,
					     false,  false, false,1180);
  if (thePieces[PHOENIX] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrPHOENIX] = new  PieceCharacteristics(PrPHOENIX, NONE,
			   12,  12, 12, 12,12,12,12,12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false,1700);
  if (thePieces[PrPHOENIX] == NULL) cerr << "Piece is null" << endl;
  thePieces[BISHOP] = new  PieceCharacteristics(BISHOP, PrBISHOP,
			  0, 0, 0, 0, 12, 12, 12, 12,
			  false, false, false, false,  false,  false,  false,  false,
			  false, false, false, false, false, false, false, false,
					     false,  false, false,1080);
  if (thePieces[BISHOP] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrBISHOP] = new  PieceCharacteristics(PrBISHOP, NONE,
			   1,  1, 1, 1,12,12,12,12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false,1150);
  if (thePieces[PrBISHOP] == NULL) cerr << "Piece is null" << endl;
  thePieces[SIDEMOVER] = new  PieceCharacteristics(SIDEMOVER, FREEBOAR,
			  1, 1,12,12, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 850);
  if (thePieces[SIDEMOVER] == NULL) cerr << "Piece is null" << endl;
  thePieces[FREEBOAR] = new  PieceCharacteristics(FREEBOAR, NONE,
			  0, 0,12,12,12,12,12,12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 1350);
  if (thePieces[FREEBOAR] == NULL) cerr << "Piece is null" << endl;
  thePieces[VERTICALMOVER] = new  PieceCharacteristics(VERTICALMOVER, FLYINGOX,
			  12, 12,1,1, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 1050);
  if (thePieces[VERTICALMOVER] == NULL) cerr << "Piece is null" << endl;
  thePieces[FLYINGOX] = new  PieceCharacteristics(FLYINGOX, NONE,
			  12, 12,0,0,12,12,12,12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 1400);
  if (thePieces[FLYINGOX] == NULL) cerr << "Piece is null" << endl;
  thePieces[ROOK] = new  PieceCharacteristics(ROOK, PrROOK,
			  12, 12,12,12, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 1200);
  if (thePieces[ROOK] == NULL) cerr << "Piece is null" << endl;
  thePieces[PrROOK] = new  PieceCharacteristics(PrROOK, NONE,
			  12, 12,12,12,1,1,1,1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 1260);
  if (thePieces[PrROOK] == NULL) cerr << "Piece is null" << endl;
  thePieces[HORSE] = new  PieceCharacteristics(HORSE, FALCON,
			  1, 1,1,1, 12, 12, 12, 12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 1500);
  if (thePieces[HORSE] == NULL) cerr << "Piece is null" << endl;
  thePieces[FALCON] = new  PieceCharacteristics(FALCON, NONE,
			  0, 12,12,12,12,12,12,12,
			  false, false, false, false, false, false, false, false,
			  true, false, false, false, false, false, false, false,
					     false, false, false, 1600);
  if (thePieces[FALCON] == NULL) cerr << "Piece is null" << endl;
  thePieces[DRAGON] = new  PieceCharacteristics(DRAGON, EAGLE,
			  12, 12,12,12, 1, 1, 1, 1,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false, 1550);
  if (thePieces[DRAGON] == NULL) cerr << "Piece is null" << endl;
  thePieces[EAGLE] = new  PieceCharacteristics(EAGLE, NONE,
			  12, 12,12,12,0,0,12,12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, true, true, false, false,
					     false, false, false, 1650);
  if (thePieces[EAGLE] == NULL) cerr << "Piece is null" << endl;
  thePieces[FREEKING] = new  PieceCharacteristics(FREEKING, NONE,
			   12,  12, 12, 12,12,12,12,12,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
					     false, false, false,1700);
  if (thePieces[FREEKING] == NULL) cerr << "Piece is null" << endl;
  thePieces[LION] = new  PieceCharacteristics(LION, NONE,
			    0,   0,  0,  0, 0, 0, 0, 0,
			  false, false, false, false, false, false, false, false,
			  false, false, false, false, false, false, false, false,
			  true, false, false,4000);
  if (thePieces[LION] == NULL) cerr << "Piece is null" << endl;
}

Board::~Board()
{
  
  if ((positions)[0]) delete (positions)[0];
  else cerr << "positions[0] is null" << endl;
  if ((positions)[1]) delete (positions)[1];
  else cerr << "positions[0] is null" << endl;

  // now delete the cells

  for (jint rank=0; rank < 12; rank++)
    for (jint file=11; file >= 0; file--) {
      if (cells[file][rank]) delete cells[file][rank];
      else cerr << "cell is null" << endl;
    }

  // now delete the pieces

  if (thePieces[PAWN]) delete thePieces[PAWN];
  else cerr << "PAWN is null" << endl;
  if (thePieces[TOKIN]) delete thePieces[TOKIN];	
  else cerr << "TOKIN is null" << endl;
  if (thePieces[GOBETWEEN]) delete thePieces[GOBETWEEN];
  else cerr << "GOBETWEEN is null" << endl;
  if (thePieces[PrGOBETWEEN]) delete thePieces[PrGOBETWEEN];
  else cerr << "PrGOBETWEEN is null" << endl;
  if (thePieces[COPPER]) delete thePieces[COPPER];
  else cerr << "COPPER is null" << endl;
  if (thePieces[PrCOPPER]) delete thePieces[PrCOPPER];
  else cerr << "PrCOPPER is null" << endl;
  if (thePieces[SILVER]) delete thePieces[SILVER];
  else cerr << "SILVER is null" << endl;
  if (thePieces[PrSILVER]) delete thePieces[PrSILVER];
  else cerr << "PrSILVER is null" << endl;
  if (thePieces[GOLD]) delete thePieces[GOLD];
  else cerr << "GOLD is null" << endl;
  if (thePieces[PrGOLD]) delete thePieces[PrGOLD];
  else cerr << "PrGOLD is null" << endl;
  if (thePieces[LEOPARD]) delete thePieces[LEOPARD];
  else cerr << "LEOPARD is null" << endl;
  if (thePieces[PrLEOPARD]) delete thePieces[PrLEOPARD];
  else cerr << "PrLEOPARD is null" << endl;
  if (thePieces[DRUNKELEPHANT]) delete thePieces[DRUNKELEPHANT];
  else cerr << "DRUNKELEPHANT is null" << endl;
  if (thePieces[PrDRUNKELEPHANT]) delete thePieces[PrDRUNKELEPHANT];
  else cerr << "PrDRUNKELEPHANT is null" << endl;
  if (thePieces[KING]) delete thePieces[KING];
  else cerr << "KING is null" << endl;
  if (thePieces[LANCE]) delete thePieces[LANCE];
  else cerr << "LANCE is null" << endl;
  if (thePieces[WHITEHORSE]) delete thePieces[WHITEHORSE];
  else cerr << "WHITEHORSE is null" << endl;
  if (thePieces[REVERSECHARIOT]) delete thePieces[REVERSECHARIOT];
  else cerr << "REVERSECHARIOT is null" << endl;
  if (thePieces[WHALE]) delete thePieces[WHALE];
  else cerr << "WHALE is null" << endl;
  if (thePieces[BLINDTIGER]) delete thePieces[BLINDTIGER];
  else cerr << "BLINDTIGER is null" << endl;
  if (thePieces[FLYINGSTAG]) delete thePieces[FLYINGSTAG];
  else cerr << "FLYINGSTAG is null" << endl;  
  if (thePieces[KYLIN]) delete thePieces[KYLIN];
  else cerr << "KYLIN is null" << endl;
  if (thePieces[PrKYLIN]) delete thePieces[PrKYLIN];
  else cerr << "KYLIN is null" << endl;
  if (thePieces[PHOENIX]) delete thePieces[PHOENIX];
  else cerr << "PHOENIX is null" << endl;
  if (thePieces[PrPHOENIX]) delete thePieces[PrPHOENIX];
  else cerr << "PrPHOENIX is null" << endl;
  if (thePieces[BISHOP]) delete thePieces[BISHOP];
  else cerr << "BISHOP is null" << endl;
  if (thePieces[PrBISHOP]) delete thePieces[PrBISHOP];
  else cerr << "PrBISHOP is null" << endl;
  if (thePieces[SIDEMOVER]) delete  thePieces[SIDEMOVER];
  else cerr << "SIDEMOVER is null" << endl;
  if (thePieces[FREEBOAR]) delete thePieces[FREEBOAR];
  else cerr << "FREEBOAR is null" << endl;
  if (thePieces[VERTICALMOVER]) delete thePieces[VERTICALMOVER];
  else cerr << "VERTICALMOVER is null" << endl;
  if (thePieces[FLYINGOX]) delete thePieces[FLYINGOX];
  else cerr << "FLYINGOX is null" << endl;
  if (thePieces[ROOK]) delete thePieces[ROOK];
  else cerr << "ROOK is null" << endl;
  if (thePieces[PrROOK]) delete thePieces[PrROOK];
  else cerr << "PrROOK is null" << endl;
  if (thePieces[HORSE]) delete thePieces[HORSE];
  else cerr << "HORSE is null" << endl;
  if (thePieces[FALCON]) delete thePieces[FALCON];
  else cerr << "FALCON is null" << endl;
  if (thePieces[DRAGON]) delete thePieces[DRAGON];
  else cerr << "DRAGON is null" << endl;
  if (thePieces[EAGLE]) delete thePieces[EAGLE];
  else cerr << "EAGLE is null" << endl;
  if (thePieces[FREEKING]) delete thePieces[FREEKING];
  else cerr << "FREEKING is null" << endl;
  if (thePieces[LION]) delete thePieces[LION];
  else cerr << "LION is null" << endl;
}

INLINE void Board::clearLighting()
{
  for (jint rank=0; rank < 12; rank++)
    for (jint file=11; file >= 0; file--) {
      (*cells[file][rank]).clearLighting();
    }
}

INLINE void Board::setLighting(Coordinates const &c)
{ 
  clearLighting();

  jint X=c.x();
  jint Y=c.y();
  Cell *thisCell=cells[X][Y];
  Piece *thisPiece=thisCell->getPiece();

  // first set the lighting for regular moves
  for (Directions dir=NORTH; dir <= SOUTHEAST; dir=Directions(dir+1)) {
    jint range=thisPiece->getRange(dir);    
    Coordinates tempCoords(c);
    if (range > 0) for (jint r=1; r <=range; r++) {
      tempCoords.inc(dir,1,thisCell->isBlack());
      
      jint tempRank=tempCoords.y();
      jint tempFile=tempCoords.x();
      if (tempFile < 0 || tempFile > 11) break;
      if (tempRank < 0 || tempRank > 11) break;
      
      Cell *cellToHighlight=cells[tempFile][tempRank];
      Piece *tempPiece=cellToHighlight->getPiece();
      if (tempPiece && (tempPiece->isBlack() == thisCell->isBlack()))
	break; // blocked by our own piece
      else if (!tempPiece) cellToHighlight->setLighting(STEP); // clear square
      else { // opposing piece
	cellToHighlight->setLighting(STEP_CAPTURE);
	break; // that's the end
      }
    }

    bool jump=thisPiece->getJump(dir);    
    if (jump) do {
      Coordinates tempCoords2(c);
      tempCoords2.inc(dir,2,thisCell->isBlack());

      jint tempRank=tempCoords2.y();
      jint tempFile=tempCoords2.x();
      if (tempFile < 0 || tempFile > 11) break;
      if (tempRank < 0 || tempRank > 11) break;
      
      Cell *cellToHighlight=cells[tempFile][tempRank];
      Piece *tempPiece=cellToHighlight->getPiece();
      if (tempPiece && (tempPiece->isBlack() == thisCell->isBlack()))
	break; // our piece
      else if (!tempPiece) cellToHighlight->setLighting(JUMP); // clear square
      else { // opposing piece
	cellToHighlight->setLighting(JUMP_CAPTURE);
	break; // that's the end
      }
    } while (false);

    int square=0;
    bool lion=thisPiece->getLion(dir);    
    Coordinates tempCoords2(c);
    if (lion) while (square < 2) {
      
      tempCoords2.inc(dir,1,thisCell->isBlack());

      jint tempRank=tempCoords2.y();
      jint tempFile=tempCoords2.x();

      if (tempFile < 0 || tempFile > 11) {
	square++;
	continue;
      }
      if (tempRank < 0 || tempRank > 11) {
	square++;
	continue;
      }
      
      Cell *cellToHighlight=cells[tempFile][tempRank];
      Piece *tempPiece=cellToHighlight->getPiece();
      Lighting ordinary, capture; 
      if (square == 1) { // Lion B
	ordinary=LION_B;
	capture=LION_B_CAPTURE;
      }
      else { // Lion A
	ordinary=LION_A;
	capture=LION_A_CAPTURE;
      }

      if (tempPiece && (tempPiece->isBlack() == thisCell->isBlack())) {
	square++;
	continue;
      }
      else if (!tempPiece) {
	cellToHighlight->setLighting(ordinary); // empty square
      }
      else { // opposing piece
	cellToHighlight->setLighting(capture);
      }
      square++;
    }

  }

  // now do Lion Power

  if (thisPiece->getLionPower()) {
    for (jint rank=Y-2; rank <= Y+2; rank++) {
      if (rank < 0 || rank > 11) continue;
      for (jint file=X-2; file <= X+2; file++) {
	if (rank == Y && file == X) thisCell->setLighting(CLEAR); // origin
        else if (file < 0 || file > 11) continue;
	else if (abs(rank-Y) < 2 && abs(file-X) < 2) { // Lion A
	  Cell *cell=cells[file][rank];
	  Piece *piece=cell->getPiece();
	  if (!piece) cell->setLighting(LION_A);
	  else if (piece->isBlack() != thisCell->isBlack()) cell->setLighting(LION_A_CAPTURE);
	}
	else { // Lion B
	  Cell *cell=cells[file][rank];
	  Piece *piece=cell->getPiece();
	  if (!piece) cell->setLighting(LION_B);
	  else if (piece->isBlack() != thisCell->isBlack()) cell->setLighting(LION_B_CAPTURE);
	}
      }
    }
  }
}
//try inline
bool Board::isLegal(Move *currentMove,const Move &previousMove)
{
  // this is only called at stage 1

  const Coordinates &destination=currentMove->getStage1();
  const Cell &thisCell=*(cells[destination.x()][destination.y()]);
  Lighting light=thisCell.getLighting();
  
  if ( light == CLEAR ) return false;
  if ( light == LION_A || light == LION_A_CAPTURE ) currentMove->setAnotherMovePossible();
  if (currentMove->isLionTarget()) {
      if (currentMove->isLionSource()) {
	if ( light == LION_A_CAPTURE ) return true;
	// logically, must be a lion B capture, so temporarily remove the source Lion 
	const Coordinates &source=currentMove->getSource();
	Cell &sourceCell=*(cells[source.x()][source.y()]);
	Piece *sourceLion=sourceCell.getPiece();
	sourceCell.reset();
	// is the target lion protected?
	bool protect=isSquareUnprotected(destination,!currentMove->isBlack());
	// restore the lion
	sourceCell.set(sourceLion);
	return protect;
      } // Lion Source
      else if (previousMove.wasLionCaptured()) return false;
      else return true; // free for non-lion to capture
  } // Lion Target
  else return true; // non-lion target
}

INLINE bool Board::isSquareUnprotected(const Coordinates &destination,bool colour)
{
  bool covered=false;

  Cell &theSquare=*(cells[destination.x()][destination.y()]);
  Piece *thePiece=theSquare.getPiece(); // save it
  Lighting   savedLights[12][12];
  
  for (jint rank=0; rank < 12; rank++)
    for (jint file=11; file >=0; file--)
      savedLights[file][rank]=cells[file][rank]->getLighting(); // save the lights
  theSquare.reset(); // remove the piece
  clearLighting();
  setInfluence(destination,colour);
  // if any square has non-CLEAR lighting, this square is protected
  for (jint rank=0; rank < 12; rank++)
    for (jint file=11; file >=0; file--) {
      if (cells[file][rank]->getLighting() != CLEAR) covered = true;
      cells[file][rank]->setLighting(savedLights[file][rank]); // restore the original light
    }
  theSquare.set(thePiece);

  return (!covered);
}

INLINE void Board::setInfluence(Coordinates const&c, bool colour)
{  // this is only used by isSquare unprotected
  for (jint rank=0; rank < 12; rank++)
    for (jint file=11; file >=0; file--) {
      if (rank==c.y() && file == c.x()) continue;

      Cell &theSquare=*(cells[file][rank]);
      Piece *thePiece=theSquare.getPiece(); 
      if (thePiece && thePiece->isBlack() == colour) {
	// we determine the lighting for thisPiece TOWARDS c
	// first set the lighting for regular moves
	for (Directions dir=NORTH; dir <= SOUTHEAST; dir=Directions(dir+1)) {
	  jint range=thePiece->getRange(dir);    
	  Coordinates tempCoords(file,rank);
	  if (range > 0) for (jint r=1; r <=range; r++) {
	    tempCoords.inc(dir,1,theSquare.isBlack());
	    
	    jint tempRank=tempCoords.y();
	    jint tempFile=tempCoords.x();
	    if (tempFile < 0 || tempFile > 11) break;
	    if (tempRank < 0 || tempRank > 11) break;
      
	    Cell *someCell=cells[tempFile][tempRank];
	    if (tempCoords == c) {
	      theSquare.setLighting(STEP);
	      break;
	    }
	    else if (someCell->getPiece()) break;
	  }

	  if (thePiece->getJump(dir))  do {
	    Coordinates tempCoords(file,rank);
	    tempCoords.inc(dir,2,theSquare.isBlack());

    
	    jint tempRank=tempCoords.y();
	    jint tempFile=tempCoords.x();
	    if (tempFile < 0 || tempFile > 11) break;
	    if (tempRank < 0 || tempRank > 11) break;      
	    if (tempCoords == c) {
	      theSquare.setLighting(JUMP);
	      break;
	    }
	    break; // only do it once
	  } while(true);

	  int square=0;

	  if (thePiece->getLion(dir)) { 
	    Coordinates tempCoords2(file,rank);
	    while (square < 2) {
	      tempCoords2.inc(dir,1,theSquare.isBlack());
	      
	      jint tempRank=tempCoords2.y();
	      jint tempFile=tempCoords2.x();
	      if (tempFile < 0 || tempFile > 11) {
		square++;
		continue;
	      }
	      if (tempRank < 0 || tempRank > 11) {
		square++;
		continue;
	      }
	      if (tempCoords2 == c) theSquare.setLighting(LION_A); // any light will do
	      square++;
	    }
	  }
	}
	// now for Lion Power
	if (thePiece->getLionPower())
	  for (jint lionRank=rank-2; lionRank <= rank+2; lionRank++) {
            if (lionRank < 0 || lionRank > 11) continue; // might come back on to the board yet
	    for (jint lionFile=file-2; lionFile <= file+2; lionFile++) {
	      if (lionFile < 0 || lionFile > 11) continue; // might come back on to the board yet
	      Coordinates tempCoords(lionFile,lionRank);
	      if (lionFile == file && lionRank == rank) continue; //ignore the origin
	      if (tempCoords == c) theSquare.setLighting(LION_A); // any light will do
	    }
	  }
      }
    }
}

void INLINE Board::adjustCPsAfterCapture(Piece* target)
{
  bool colour = target->isBlack();
  if (target->isCrownPrince()) crownPrinces[colour]-=1;
  piecesOnBoard[colour]-=1;  
}

void INLINE Board::adjustCPsAfterRestoration(Piece* target)
{
  bool colour = target->isBlack();
  if (target->isCrownPrince()) crownPrinces[colour]+=1;
  piecesOnBoard[colour]+=1;  
}

void INLINE Board::adjustCPsAfterPromotion(Piece* target)
{
  bool colour = target->isBlack();
  if (target->isCrownPrince()) crownPrinces[colour]+=1;
}

void INLINE Board::adjustCPsBeforeDemotion(Piece* target)
{
  bool colour = target->isBlack();
  if (target->isCrownPrince()) crownPrinces[colour]-=1;
}

INLINE bool Board::performMove(Move&move,Targets&t, bool repetitions)
{
  //  char buffer[30];
  // move.notate(buffer);
  //cout << "performing move: " << buffer << endl;
  
  Coordinates targetCoordinates=move.getStage1();
  Cell &targetCell=*cells[targetCoordinates.x()][targetCoordinates.y()];
  Piece *targetPiece=targetCell.getPiece();
  jint squares=move.squares();
  bool ok;

  if (move.wasPassOrIgui()) {
    if (move.wasFirstCaptured()) { // igui - no promotion complications
      t.target.set(targetPiece);
      adjustCPsAfterCapture(targetPiece);
      targetCell.reset();
      if (targetPiece) delete targetPiece;
      else cerr << "targetPiece is null" << endl;
      if (repetitions) ok=addPosition(move.isBlack());
      else return true;
      return ok;
    }
    else { // pass - can't happen right now, as we don't generate any passes
      if (repetitions) ok=addPosition(move.isBlack()); // even a pass affects the repetition rule
      else return true;
      return ok;
    }
  }

  // ordinary moves
  Coordinates sourceCoordinates=move.getSource();
  Cell &sourceCell=*cells[sourceCoordinates.x()][sourceCoordinates.y()];
  Piece *sourcePiece=sourceCell.getPiece();
  t.source.set(sourcePiece);
  if (move.wasFirstCaptured()) {
    t.target.set(targetPiece);
    adjustCPsAfterCapture(targetPiece);
    targetCell.reset();
    if (targetPiece) delete targetPiece;
    else cerr << "targetPiece is null" << endl;
  }
  if (squares==2) { // second move
    Coordinates target2Coordinates=move.getStage2();
    Cell &target2Cell=*cells[target2Coordinates.x()][target2Coordinates.y()];
    Piece *target2Piece=target2Cell.getPiece();
    if (move.wasSecondCaptured()) {
      t.target2.set(target2Piece);
      adjustCPsAfterCapture(target2Piece);
      if (target2Piece) delete target2Piece;
      else cerr << "target2Piece is null" << endl;
    }
    target2Cell.set(sourcePiece);
  }
  else { // single move
    if (move.wasPromoted()) {
      sourcePiece->promote(thePieces);
      adjustCPsAfterPromotion(sourcePiece);
    }
    targetCell.set(sourcePiece);
  }
  sourceCell.reset();
  // not promoted adjustments
  if (move.isOK()) {
    sourcePiece->setOK(true);
    //char buffer[30];
    //move.notate(buffer);
    //cerr << "Promotion was OK for " << buffer << endl; 
  }
  if (move.wasDelayed()) {
    sourcePiece->setDelayed(true);
    //char buffer[30];
    //move.notate(buffer);
    //cerr << "Promotion was delayed for " << buffer << endl; 
  }
  
  if (repetitions) ok=addPosition(move.isBlack()); // even a pass affects the repetition rule
  else return true;
  return ok;
}


void Board::backOutMove(Move &move, Targets&t, bool repetitions)
{
  // char buffer[30];
  //move.notate(buffer);
  //cerr << "restoring move: " << buffer << endl;

  if (repetitions) restorePosition(move.isBlack());          // repetition rule
  jint squares=move.squares();  
  Coordinates targetCoordinates=move.getStage1();
  Cell &targetCell=*cells[targetCoordinates.x()][targetCoordinates.y()];
  Coordinates target2Coordinates=move.getStage2();
  Cell &target2Cell=*cells[target2Coordinates.x()][target2Coordinates.y()];
  Coordinates sourceCoordinates=move.getSource();
  Cell &sourceCell=*cells[sourceCoordinates.x()][sourceCoordinates.y()];
  Piece *oldSourcePiece=NULL;
  Piece *sourcePiece=NULL;
  if (squares==1)
    oldSourcePiece=targetCell.getPiece();
  else oldSourcePiece=target2Cell.getPiece();


  if (move.wasPassOrIgui()) {
    if (move.wasFirstCaptured()) {
      Piece *targetPiece=new Piece();
      if (targetPiece == NULL) cerr << "targetPiece allocation failed" << endl;
      t.target.restore(targetPiece);
      adjustCPsAfterRestoration(targetPiece);
      targetCell.set(targetPiece);          // restore captured piece
    }
    return;                                 // nought else to do
  }
  if (move.wasPromoted()) {
    adjustCPsBeforeDemotion(oldSourcePiece);
    if (oldSourcePiece) delete oldSourcePiece;
    else cerr << "oldSourcePiece is null" << endl;
    sourcePiece=new Piece();
    if (sourcePiece == NULL) cerr << "sourcePiece allocation failed" << endl;
    t.source.restore(sourcePiece);
    sourcePiece->demote(thePieces);
  }
  else if (squares == 2) { // second move
    Coordinates target2Coords=move.getStage2();
    Cell &target2Cell=*cells[target2Coords.x()][target2Coords.y()];
    if (move.wasSecondCaptured()) {
      Piece *target2Piece=new Piece();
      if (target2Piece == NULL) cerr << "target2Piece allocation failed" << endl;
      t.target2.restore(target2Piece);
      adjustCPsAfterRestoration(target2Piece);
      target2Cell.set(target2Piece);
    }
    else target2Cell.reset();
  }

    // common to single and double moves

  if (move.wasFirstCaptured()) {
    Piece *targetPiece=new Piece();
    if (targetPiece == NULL) cerr << "targetPiece allocation failed" << endl;
    t.target.restore(targetPiece);
    adjustCPsAfterRestoration(targetPiece);
    targetCell.set(targetPiece);            // restore captured piece
  }
  else  targetCell.reset();

  if (!move.wasPromoted()) {
    if (oldSourcePiece) delete oldSourcePiece;
    else cerr << "oldSourcePiece is null" << endl;
    sourcePiece=new Piece();
    if (sourcePiece == NULL) cerr << "sourcePiece allocation failed" << endl;
    t.source.restore(sourcePiece);
  }
  sourceCell.set(sourcePiece);
}
//compute the Forsyth description of the board
INLINE Forsyth* Board::forsyth()
{
  Forsyth *f=new Forsyth();

  for (int rank=0; rank < 12; rank++)
    for (int file=11; file >= 0; file--) {
      Cell &cell=*cells[file][rank];
      Piece *piece=cell.getPiece();
      if (!piece) f->set(file,rank,NONE);
      else {
	bool colour=piece->isBlack();
	if (colour) f->set(file,rank,jbyte(piece->getAbbrev()) | 0x80);
	else f->set(file,rank,jbyte(piece->getAbbrev()));
      }
    }
  return f;
}
/*
#include <iomanip>
void Board::describeBoard()
{
  for (jint rank=0; rank < 12; rank++) {
    for (jint file=0; file<12; file++) {
      Cell &thisCell=getCell(file,rank);
      Piece *piece=thisCell.getPiece();
      if (piece) {
	char ca[4];
	char *casedAbbrev=&ca[0];
	string abbrev=piece->getAbbrevString();
	if (piece->isBlack()) {
	  const char *lower=abbrev.c_str();
	  while(char ch=*lower++) {
	    *casedAbbrev++=tolower(ch);
	  }
	}
	else {
	  const char *upper=abbrev.c_str();
	  while(char ch=*upper++) {
	    *casedAbbrev++=toupper(ch);
	  }
	}
	*casedAbbrev='\0';
	cout << " " << ca;
        int len=strlen(ca);
        for (int spaces=4-len; spaces>0; spaces--) cout << " ";
      }
      else cout << "  #  ";
    }
    cout << endl;
  }
}*/

static bool isLower(const char *);
static PieceType stringToPieceType(const char *);

INLINE void Board::addPosition(string str, bool col) {
  Forsyth *f=new Forsyth();

  int file=11;
  int rank=-1;
  int index=0;
  int len=str.length();
  int value=-1;
  int fieldIndex=-1;
  PieceType piece=NONE;
  bool seeingValue=false;
  bool seeingPiece=false;

  while (index < len) {
    char ch=str[index++];
    //cout << ch << endl;
    if (ch == '/') {
      if (seeingValue) {
	//        cout << value << " spaces," << endl;
	for (; value > 0; value --) {
	  f->set(file,rank,NONE);
	  file--;
	}
	//cout << " file is now " << file << " (should be -1)" << endl;
      }
      else if (seeingPiece) {
        string pieceString=str.substr(fieldIndex,index-fieldIndex-1);
	bool colour=isLower(pieceString.c_str());
	piece=stringToPieceType(pieceString.c_str());
        //cout << "piece is " << piece << ", colour is " << colour << endl;
        //cout << "file is " << file << ", rank is " << rank << endl;
	if (colour) f->set(file--,rank,jbyte(piece | 0x80));
	else f->set(file--,rank,jbyte(piece));
      }
      value=0;
      file=11;
      rank++;
      fieldIndex=index; // next character
      seeingValue=false;
      seeingPiece=false;
    }
    else if (ch == ',') {
      if (seeingValue) {
        //cout << value << " spaces," << endl;
	for (; value > 0; value --) {
	  f->set(file,rank,NONE);
	  file--;
	}
	//cout << " file is now " << file << endl;
      }
      else if (seeingPiece) {
        string pieceString=str.substr(fieldIndex,index-fieldIndex-1);
	bool colour=isLower(pieceString.c_str());
	piece=stringToPieceType(pieceString.c_str());
	//    cout << "piece is " << piece << ", colour is " << colour << endl;
        //cout << "file is " << file << ", rank is " << rank << endl;
	if (colour) f->set(file--,rank,jbyte(piece | 0x80));
	else f->set(file--,rank,jbyte(piece));
      }
      value=0;
      fieldIndex=index; // next character
      seeingValue=false;
      seeingPiece=false;
    }
    else if (isdigit(ch)) {
      seeingValue=true;
      value*=10;
      value+=int(ch) - int('0');
    }
    else seeingPiece=true;
  }
  //f->describe(this);
  positions[col]->insert(*f); 
  delete f;
}

static INLINE bool isLower(const char *str)
{
  char ch;
  while((ch=*str++)) {
    if (islower(ch)) return true;
  }
  return false;
}

static INLINE PieceType stringToPieceType(const char *str)
{
  PieceType moveAbbrev=NONE;
  char abbrevStr[4];
  char *abb=&abbrevStr[0];
  char ch;

  while((ch=*str++))
    *abb++ = toupper(ch);
  *abb='\0';

  if (strcmp(abbrevStr,"P") == 0) moveAbbrev=PAWN;
  else if (strcmp(abbrevStr,"+P") == 0) moveAbbrev=TOKIN;
  else if (strcmp(abbrevStr,"GB") == 0) moveAbbrev=GOBETWEEN;
  else if (strcmp(abbrevStr,"+GB") == 0) moveAbbrev=PrGOBETWEEN;
  else if (strcmp(abbrevStr,"C") == 0) moveAbbrev=COPPER;
  else if (strcmp(abbrevStr,"+C") == 0) moveAbbrev=PrCOPPER;
  else if (strcmp(abbrevStr,"S") == 0) moveAbbrev=SILVER;
  else if (strcmp(abbrevStr,"+S") == 0) moveAbbrev=PrSILVER;
  else if (strcmp(abbrevStr,"G") == 0) moveAbbrev=GOLD;
  else if (strcmp(abbrevStr,"+G") == 0) moveAbbrev=PrGOLD;
  else if (strcmp(abbrevStr,"FL") == 0) moveAbbrev=LEOPARD;
  else if (strcmp(abbrevStr,"+FL") == 0) moveAbbrev=PrLEOPARD;
  else if (strcmp(abbrevStr,"DE") == 0) moveAbbrev=DRUNKELEPHANT;
  else if (strcmp(abbrevStr,"+DE") == 0) moveAbbrev=PrDRUNKELEPHANT;
  else if (strcmp(abbrevStr,"K") == 0) moveAbbrev=KING;
  else if (strcmp(abbrevStr,"L") == 0) moveAbbrev=LANCE;
  else if (strcmp(abbrevStr,"+L") == 0) moveAbbrev=WHITEHORSE;
  else if (strcmp(abbrevStr,"RC") == 0) moveAbbrev=REVERSECHARIOT;
  else if (strcmp(abbrevStr,"+RC") == 0) moveAbbrev=WHALE;
  else if (strcmp(abbrevStr,"BT") == 0) moveAbbrev=BLINDTIGER;
  else if (strcmp(abbrevStr,"+BT") == 0) moveAbbrev=FLYINGSTAG;
  else if (strcmp(abbrevStr,"KY") == 0) moveAbbrev=KYLIN;
  else if (strcmp(abbrevStr,"+KY") == 0) moveAbbrev=PrKYLIN;
  else if (strcmp(abbrevStr,"PH") == 0) moveAbbrev=PHOENIX;
  else if (strcmp(abbrevStr,"+PH") == 0) moveAbbrev=PrPHOENIX;
  else if (strcmp(abbrevStr,"B") == 0) moveAbbrev=BISHOP;
  else if (strcmp(abbrevStr,"+B") == 0) moveAbbrev=PrBISHOP;
  else if (strcmp(abbrevStr,"SM") == 0) moveAbbrev=SIDEMOVER;
  else if (strcmp(abbrevStr,"+SM") == 0) moveAbbrev=FREEBOAR;
  else if (strcmp(abbrevStr,"VM") == 0) moveAbbrev=VERTICALMOVER;
  else if (strcmp(abbrevStr,"+VM") == 0) moveAbbrev=FLYINGOX;
  else if (strcmp(abbrevStr,"R") == 0) moveAbbrev=ROOK;
  else if (strcmp(abbrevStr,"+R") == 0) moveAbbrev=PrROOK;
  else if (strcmp(abbrevStr,"DH") == 0) moveAbbrev=HORSE;
  else if (strcmp(abbrevStr,"+DH") == 0) moveAbbrev=FALCON;
  else if (strcmp(abbrevStr,"DK") == 0) moveAbbrev=DRAGON;
  else if (strcmp(abbrevStr,"+DK") == 0) moveAbbrev=EAGLE;
  else if (strcmp(abbrevStr,"FK") == 0) moveAbbrev=FREEKING;
  else if (strcmp(abbrevStr,"LN") == 0) moveAbbrev=LION;
  else cerr << "Unknown piece type in stringToPieceType: " << abbrevStr << "###" << endl;

  return moveAbbrev;
}

void Forsyth::describe(Board*b) {
  for (int rank=0; rank < 12; rank++) {
    for (int file=11; file > -1; file--) {
      PieceCharacteristics*p=(b->getPieces())[0x7f & (forsyth[file][rank])];

      cout << int(p->getAbbrev()) << " ";
    }
    cout << endl;
  }
}

INLINE bool Board::isCheck(bool colour)
{
  // colour is the side giving check

  // can't be in check with both a Crown Prince and a King on the board
  if (getCrownPrinces(!colour) > 1) return false; 

  // locate the King or Crown Prince in trouble

  for (int rank=0; rank < 12; rank++) 
    for (int file=11; file > -1; file--) {
      Cell &cell=*cells[file][rank];
      Piece *thePiece=cell.getPiece(); 
      if (thePiece && thePiece->isBlack() != colour && thePiece->isCrownPrince()) {
	// the victim!
	return !(isSquareUnprotected(Coordinates(file,rank),colour));
      }
    }
  return true; // can't happen, but stops compiler warning
}
